<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\CronBundle\Command;

use Cyber\CronBundle\Command\CronCommand;
use Cyber\CronBundle\Component\CronTaskInterface;
use Cyber\CronBundle\Entity\CronTaskInfo;
use Cyber\CronBundle\Event\TaskFinishedEvent;
use Cyber\CronBundle\Event\TaskStartEvent;
use Cyber\CronBundle\Manager\ScheduleContext;
use Cyber\CronBundle\Manager\TaskScheduler;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Console\Tester\CommandTester;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * @covers \Cyber\CronBundle\Command\CronCommand
 */
class CronCommandTest extends TestCase
{
    /**
     * @var MockObject|TaskScheduler
     */
    private $scheduler;

    /**
     * @var EventDispatcherInterface|MockObject
     */
    private $dispatcher;

    /**
     * @var CronCommand
     */
    private $command;

    public function testNoTasksCommand(): void
    {
        $commandTester = new CommandTester($this->command);
        $commandTester->execute([]);

        $output = $commandTester->getDisplay();
        $this->assertContains('No pending tasks.', $output);
    }

    public function testWithTask(): void
    {
        $task = $this->getMockBuilder(CronTaskInterface::class)->getMock();
        $meta = $this->getMockBuilder(CronTaskInfo::class)->getMock();

        $context = new ScheduleContext($task, $meta);
        $this->scheduler->expects(self::atLeast(1))
            ->method('nextContext')
            ->willReturnOnConsecutiveCalls($context);

        $this->dispatcher->expects(self::exactly(2))
            ->method('dispatch')
            ->withConsecutive([
                self::callback(function ($event) use ($context) {
                    self::assertInstanceOf(TaskStartEvent::class, $event);
                    self::assertSame($context, $event->toContext());

                    return true;
                }),
            ], [
                self::callback(function ($event) use ($context) {
                    self::assertInstanceOf(TaskFinishedEvent::class, $event);
                    self::assertSame($context, $event->toContext());

                    return true;
                }),
            ])->willReturnArgument(0);

        $commandTester = new CommandTester($this->command);
        $commandTester->execute([]);

        $output = $commandTester->getDisplay();
        $this->assertContains('SUCCESS', $output);
    }

    protected function setUp(): void
    {
        $this->scheduler  = $this->getMockBuilder(TaskScheduler::class)->disableOriginalConstructor()->getMock();
        $this->dispatcher = $this->getMockBuilder(EventDispatcherInterface::class)->getMock();
        $this->command    = new CronCommand($this->scheduler, $this->dispatcher);
    }
}
