<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Command;

use Cyber\CronBundle\Event\TaskFinishedEvent;
use Cyber\CronBundle\Event\TaskStartEvent;
use Cyber\CronBundle\Manager\TaskRunner;
use Cyber\CronBundle\Manager\TaskScheduler;
use Doctrine\DBAL\Exception\ConnectionException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

class CronCommand extends Command
{
    /**
     * @var TaskScheduler
     */
    private $scheduler;

    /**
     * @var EventDispatcherInterface
     */
    private $dispatcher;

    public function __construct(TaskScheduler $scheduler, EventDispatcherInterface $eventDispatcher)
    {
        parent::__construct();
        $this->scheduler  = $scheduler;
        $this->dispatcher = $eventDispatcher;
    }

    protected function configure(): void
    {
        $this->setName('cyber:cron:run')
            ->setDescription('Runs the next scheduled task');
    }

    /**
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io      = new SymfonyStyle($input, $output);
        $runner  = new TaskRunner();
        $context = $this->scheduler->nextContext();

        if (!$context) {
            $io->success('No pending tasks.');

            return 0;
        }

        $hadFailures = false;

        do {
            $task = $context->getTask();
            $meta = $context->getMeta();

            $io->title($meta->getServiceId());

            try {
                $io->writeln(\sprintf('<info>Schedule: <comment>%s</></>', $task->getSchedule()));
                $event = new TaskStartEvent($context);
                $this->dispatcher->dispatch($event);
                if ($event->isCancelled()) {
                    $io->writeln('<info>Result:   <comment>CANCELLED</></>');
                    $this->scheduler->completed($context, 'Execution cancelled via event');
                    continue;
                }
                $message = $runner->run($task);
                $io->writeln('<info>Result:   <comment>SUCCESS</></>');
                if ($message && $output->isVerbose()) {
                    $io->section('Output');
                    $io->writeln($message);
                }
            } catch (\Throwable $ex) {
                $io->writeln('<info>Result:   <comment>FAILURE</></>');
                $io->writeln('<error>Error:    ' . $ex->getMessage() . '"</error>');
                if (!$this->isTransientException($ex)) {
                    $context->markFailed($ex);
                }
                $message     = $ex->getMessage();
                $hadFailures = true;
            }
            $this->dispatcher->dispatch(new TaskFinishedEvent($context));
            $this->scheduler->completed($context, $message);
        } while ($context = $this->scheduler->nextContext());

        $hadFailures ?
            $io->warning('Some tasks failed') : $io->success('All pending tasks completed.');

        return 0;
    }

    /**
     * Check if exception is transient.
     *
     * Some exceptions such as connection can occur randomly are are most frequently just temporary and
     * can be ignored.
     *
     * @param \Throwable $exception
     *
     * @return bool
     */
    private function isTransientException(\Throwable $exception): bool
    {
        // TODO maybe this should be moved to a config section
        return $exception instanceof ConnectionException;
    }
}
