<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Entity;

use DateTime;
use DateTimeInterface;
use Doctrine\ORM\Mapping as ORM;
use Doctrine\ORM\Mapping\MappedSuperclass;

/**
 * CronTaskInfo.
 *
 * @MappedSuperclass
 *
 * @SuppressWarnings(PHPMD.BooleanGetMethodName)
 */
abstract class CronTaskInfo
{
    /**
     * @var class-string
     *
     * @ORM\Column(type="string", length=100, unique=true)
     */
    protected $serviceId;

    /**
     * @var null|DateTimeInterface
     *
     * @ORM\Column(type="datetime", nullable=true)
     */
    protected $lastRun;

    /**
     * @var DateTimeInterface
     *
     * @ORM\Column(type="datetime")
     */
    protected $nextRun;

    /**
     * @var null|int
     *
     * @ORM\Column(type="integer", nullable=true)
     */
    protected $executingPid;

    /**
     * @var bool
     *
     * @ORM\Column(type="boolean", options={"default": false})
     */
    protected $isDisabled = false;

    /**
     * @var string
     *
     * @ORM\Column(type="text")
     */
    protected $message;

    /**
     * @var float
     *
     * @ORM\Column(type="decimal", scale=2, precision=5)
     */
    protected $progress;

    public function __construct(DateTimeInterface $nextRun = null)
    {
        $this->nextRun  = $nextRun ?? new DateTime();
        $this->message  = '';
        $this->progress = 0;
    }

    /**
     * Get serviceId.
     *
     * @return class-string
     */
    public function getServiceId(): string
    {
        return $this->serviceId;
    }

    /**
     * Set serviceId.
     *
     * @param class-string $serviceId
     *
     * @return CronTaskInfo
     */
    public function setServiceId(string $serviceId): self
    {
        $this->serviceId = $serviceId;

        return $this;
    }

    /**
     * Get lastRun.
     *
     * @return null|DateTimeInterface
     */
    public function getLastRun(): ?DateTimeInterface
    {
        return $this->lastRun;
    }

    /**
     * Set lastRun.
     *
     * @param DateTime $lastRun
     *
     * @return CronTaskInfo
     */
    public function setLastRun(DateTimeInterface $lastRun): self
    {
        $this->lastRun = $lastRun;

        return $this;
    }

    /**
     * @return DateTimeInterface
     */
    public function getNextRun(): DateTimeInterface
    {
        return $this->nextRun;
    }

    /**
     * @param DateTimeInterface $nextRun
     *
     * @return CronTaskInfo
     */
    public function setNextRun(DateTimeInterface $nextRun): self
    {
        $this->nextRun = $nextRun;

        return $this;
    }

    /**
     * Marks task as running.
     *
     * Do NOT call this in the context of cron task execution. The task may clear/modify entity manager causing these
     * change to be lost or fail to persist.
     */
    public function markRunning(int $pid, DateTimeInterface $altStartTime = null): self
    {
        $this->executingPid = $pid;
        $this->lastRun      = $altStartTime ?? new DateTime();

        return $this;
    }

    public function isRunning(): bool
    {
        return null !== $this->executingPid;
    }

    /**
     * Marks task as finished.
     *
     * Do NOT call this in the context of cron task execution. The task may clear/modify entity manager causing these
     * change to be lost or fail to persist.
     */
    public function markFinished(DateTimeInterface $nextExecution): self
    {
        $this->executingPid = null;
        $this->nextRun      = $nextExecution;

        return $this;
    }

    /**
     * Set isDisabled.
     *
     * @param bool $isDisabled
     *
     * @return CronTaskInfo
     */
    public function disable($isDisabled): self
    {
        $this->isDisabled = $isDisabled;

        return $this;
    }

    /**
     * Get isDisabled.
     *
     * @return bool
     */
    public function isDisabled(): bool
    {
        return $this->isDisabled;
    }

    /**
     * @param null|DateTimeInterface $atTime used to set specific time, if null current time is used
     */
    public function shouldRun(DateTimeInterface $atTime = null): bool
    {
        $timeNow = $atTime ? new DateTime('@' . $atTime->getTimestamp()) : new DateTime();
        $timeNow->modify('+30 seconds');

        return $this->nextRun->getTimestamp() < $timeNow->getTimestamp();
    }

    public function toMessage(): string
    {
        return $this->message;
    }

    public function updateProgress(float $completed): self
    {
        $this->progress = $completed;

        return $this;
    }

    public function toProgress(): float
    {
        return $this->progress;
    }
}
