<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\CronBundle\Manager;

use Cyber\CronBundle\Component\CronTaskInterface;
use Cyber\CronBundle\Entity\CronTaskInfo;
use Cyber\CronBundle\Manager\ScheduleContext;
use Cyber\CronBundle\Manager\TaskLocker;
use Cyber\CronBundle\Manager\TaskRegistry;
use Cyber\CronBundle\Manager\TaskScheduler;
use DateTimeInterface;
use Doctrine\ORM\EntityRepository;
use Exception;
use Generator;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

/**
 * @covers \Cyber\CronBundle\Manager\TaskScheduler
 */
class TaskSchedulerTest extends TestCase
{
    /** @var TaskScheduler */
    private $instance;

    /**
     * @var MockObject|TaskRegistry
     */
    private $taskRegistry;

    /**
     * @var LoggerInterface|MockObject
     */
    private $logger;

    /**
     * @var MockObject|TaskLocker
     */
    private $locker;

    public function testSchedledIntervals(): void
    {
        $intervals = ['one' => 'two', 'three' => 'four'];
        $this->instance->setScheduleIntervals($intervals);
        $this->assertEquals($intervals, $this->instance->getScheduleIntervals());
    }

    public function testNextContext(): void
    {
        $taskMeta = $this->getMockBuilder(CronTaskInfo::class)->disableOriginalConstructor()->getMock();
        $task     = $this->getMockBuilder(CronTaskInterface::class)->disableOriginalConstructor()->getMock();
        $repo     = $this->getMockBuilder(EntityRepository::class)->disableOriginalConstructor()->getMock();
        $this->taskRegistry->expects(self::any())
            ->method('getRepository')
            ->willReturn($repo);

        $repo->expects(self::atLeast(1))
            ->method('findBy')
            ->willReturnOnConsecutiveCalls([], [$taskMeta]);

        // test no tasks found $repo call #1
        $context = $this->instance->nextContext();
        self::assertNull($context);

        $serviceId = 'App\\Task';

        $taskMeta->expects(self::once())
            ->method('shouldRun')
            ->willReturn(true);

        $taskMeta->expects(self::any())
            ->method('getServiceId')
            ->willReturn($serviceId);

        $this->logger->expects(self::once())
            ->method('debug')
            ->withConsecutive(['Locked task ' . $serviceId]);

        $this->taskRegistry->expects(self::once())
            ->method('findTask')
            ->withConsecutive([$serviceId])
            ->willReturn($task);

        $this->locker->expects(self::once())
            ->method('lockTask')
            ->willReturn(true);

        // test available taskMeta $repo call #2
        $context = $this->instance->nextContext();
        self::assertNotNull($context);
    }

    public function testNextContextCantRun(): void
    {
        $taskMeta = $this->getMockBuilder(CronTaskInfo::class)->disableOriginalConstructor()->getMock();
        $task     = $this->getMockBuilder(CronTaskInterface::class)->disableOriginalConstructor()->getMock();
        $repo     = $this->getMockBuilder(EntityRepository::class)->disableOriginalConstructor()->getMock();
        $this->taskRegistry->expects(self::any())
            ->method('getRepository')
            ->willReturn($repo);

        $repo->expects(self::atLeast(1))
            ->method('findBy')
            ->willReturn([$taskMeta]);

        $serviceId = 'App\\Task';

        $taskMeta->expects(self::atLeast(1))
            ->method('shouldRun')
            ->willReturnOnConsecutiveCalls(false, true, true);

        $taskMeta->expects(self::any())
            ->method('getServiceId')
            ->willReturn($serviceId);

        $this->taskRegistry->expects(self::atLeast(1))
            ->method('findTask')
            ->willReturnOnConsecutiveCalls(null, $task);

        $this->locker->expects(self::once())
            ->method('lockTask')
            ->willReturn(false);

        $this->logger->expects(self::atLeast(1))
            ->method('debug')
            ->withConsecutive(
                [$serviceId . ' does not need to run'],
                ["Task ${serviceId} got locked by another process"]
            );

        $this->logger->expects(self::once())
            ->method('warning')
            ->withConsecutive(["Task ${serviceId} not found. Meta out of sync?"]);

        // task should not run
        $context = $this->instance->nextContext();
        self::assertNull($context);

        // no task class
        $context = $this->instance->nextContext();
        self::assertNull($context);

        // task locked by another process
        $context = $this->instance->nextContext();
        self::assertNull($context);
    }

    /**
     * @dataProvider completeData
     *
     * @param mixed $unlockParams
     */
    public function testComplete(ScheduleContext $context, ?string $message, $unlockParams): void
    {
        $this->instance->setScheduleIntervals(['always' => '1 min']);
        $this->locker->expects(self::once())
            ->method('unlockTask')
            ->with(...$unlockParams);

        $this->instance->completed($context, $message);
    }

    /**
     * @return Generator<mixed>
     */
    public function completeData(): Generator
    {
        $taskMeta = $this->getMockBuilder(CronTaskInfo::class)->disableOriginalConstructor()->getMock();
        $task     = $this->getMockBuilder(CronTaskInterface::class)->disableOriginalConstructor()->getMock();

        $task->expects(self::atLeast(1))
            ->method('getSchedule')
            ->willReturnOnConsecutiveCalls('always', 'always', 'bad', 'bad', 'always', 'bad');

        $callback = self::callback(function ($value) {
            return $value instanceof DateTimeInterface;
        });

        $successContext = new ScheduleContext($task, $taskMeta);

        $failedContext = new ScheduleContext($task, $taskMeta);
        $failedContext->markFailed(new Exception('test error'));

        //last arg [CronTaskInfo $meta, array $messages, bool $disable, ?\DateTimeInterface $nextRun]
        yield 'success' => [
            $successContext,
            null,
            [$taskMeta, [], false, $callback],
        ];
        yield 'success with message' => [
            $successContext,
            'testMsg',
            [$taskMeta, ['testMsg'], false, $callback],
        ];
        yield 'no schedule' => [
            $successContext,
            null,
            [$taskMeta, ['Could not find schedule definition for this task'], true, null],
        ];
        yield 'no schedule with message' => [
            $successContext,
            'testMsg',
            [$taskMeta, ['testMsg', 'Could not find schedule definition for this task'], true, null],
        ];
        yield 'failure exception' => [
            $failedContext,
            null,
            [$taskMeta, ['Exception: test error'], true, null],
        ];
        yield 'failure and no schedule' => [
            $failedContext,
            null,
            [$taskMeta, ['Exception: test error', 'Could not find schedule definition for this task'], true, null],
        ];
    }

    protected function setUp(): void
    {
        $this->taskRegistry = $this->getMockBuilder(TaskRegistry::class)->disableOriginalConstructor()->getMock();
        $this->locker       = $this->getMockBuilder(TaskLocker::class)->disableOriginalConstructor()->getMock();
        $this->logger       = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $this->instance     = new TaskScheduler($this->taskRegistry, $this->locker, $this->logger);
    }
}
