<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Command;

use Cyber\CronBundle\Manager\TaskRegistry;
use Cyber\CronBundle\Manager\TaskScheduler;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class ValidateCronTasksCommand extends Command
{
    protected static $defaultName = 'cyber:cron:validate';

    private $taskRegistry;

    private $scheduler;

    public function __construct(TaskRegistry $taskRegistry, TaskScheduler $scheduler)
    {
        parent::__construct();
        $this->taskRegistry = $taskRegistry;
        $this->scheduler    = $scheduler;
    }

    protected function configure(): void
    {
        $this->setDescription('Validates that tasks and configs are correct.');
    }

    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $io = new SymfonyStyle($input, $output);
        $io->title('Cyber Cron Task Validation');

        $schedules = $this->scheduler->getScheduleIntervals();
        $tasks     = $this->taskRegistry->getTaskClassMap();
        $withTasks = [];
        $missing   = [];

        foreach ($tasks as $class => $task) {
            $schedule = $task->getSchedule();
            if (!isset($schedules[$schedule])) {
                $missing[] = [$schedule, $class];
            }
            $withTasks[$schedule] = true;
        }

        $emptySchedules = \array_diff(\array_keys($schedules), \array_keys($withTasks));

        if ($emptySchedules) {
            $io->text('<info>The following schedules are empty:</>');
            $io->listing(
                \array_map(
                    function ($schedule) use ($schedules) {
                        return \sprintf('%s: %s', $schedule, $schedules[$schedule]);
                    },
                    $emptySchedules
                )
            );
        }

        if ($missing) {
            $io->section('The following schedules are missing');
            $io->table(['schedule', 'used in'], $missing);
            $io->error('Validation Failed. There are missing schedules');

            return 1;
        }

        $io->success('Validation Passed');

        return 0;
    }
}
