<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Manager;

use Cyber\CronBundle\Entity\CronTaskMetaBase;
use DateTimeImmutable;
use Doctrine\ORM\EntityRepository;

class TaskLocker
{
    private $taskRegistry;

    /**
     * @var int
     */
    private $pid;

    public function __construct(TaskRegistry $taskRegistry)
    {
        $this->taskRegistry = $taskRegistry;
        $this->pid          = \getmypid() ?: (\time() + \random_int(10000, 99999));
    }

    /**
     * @param CronTaskMetaBase       $meta
     * @param string[]               $messages
     * @param bool                   $disable
     * @param null|DateTimeImmutable $nextRun
     */
    public function unlockTask(CronTaskMetaBase $meta, array $messages, bool $disable, ?DateTimeImmutable $nextRun): void
    {
        /** @var EntityRepository<CronTaskMetaBase> $repo */
        $repo = $this->taskRegistry->getRepository();
        $qb   = $repo->createQueryBuilder('task')
            ->update()
            ->set('task.executingPid', ':nullPid')
            ->set('task.progress', $disable ? 0 : 100)
            ->set('task.message', ':message')
            ->where('task = :task')
            ->andWhere('task.executingPid = :pid')
            ->setParameter('task', $meta)
            ->setParameter('pid', $this->pid)
            ->setParameter('message', \implode('; ', $messages))
            ->setParameter('nullPid', null);

        if ($nextRun) {
            $qb
                ->set('task.nextRun', ':nextRun')
                ->setParameter('nextRun', $nextRun);
        }

        if ($disable) {
            $qb->set('task.isDisabled', true);
        }

        // !!! we want to run direct queries instead of EM/UoW updates,
        // since EM/UoW could be modified within task execution method which may cause issues (if it was cleared for ex)
        $qb->getQuery()->execute();
    }

    public function lockTask(CronTaskMetaBase $taskMeta): bool
    {
        $date = new DateTimeImmutable();

        /** @var EntityRepository<CronTaskMetaBase> $repo */
        $repo   = $this->taskRegistry->getRepository();
        $qb     = $repo->createQueryBuilder('task')
            ->update()
            ->set('task.executingPid', ':pid')
            ->set('task.progress', 0)
            ->set('task.lastRun', ':lastRun')
            ->where('task = :task')
            ->andWhere('task.executingPid IS NULL')
            ->setParameter('task', $taskMeta)
            ->setParameter('pid', $this->pid)
            ->setParameter('lastRun', $date);
        $result = $qb->getQuery()->execute();

        return 1 === $result;
    }
}
