<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\CronBundle\Manager;

use Cyber\CronBundle\Component\CronTaskInterface;
use Cyber\CronBundle\Entity\CronTaskMeta;
use Cyber\CronBundle\Manager\ScheduleContext;
use Cyber\CronBundle\Manager\TaskLocker;
use Cyber\CronBundle\Manager\TaskRegistry;
use Cyber\CronBundle\Manager\TaskScheduler;
use DateTimeInterface;
use Doctrine\ORM\EntityRepository;
use Exception;
use Generator;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Psr\Log\LoggerInterface;

/**
 * @covers \Cyber\CronBundle\Manager\TaskScheduler
 *
 * @internal
 */
class TaskSchedulerTest extends TestCase
{
    /** @var TaskScheduler */
    private $instance;

    /**
     * @var MockObject|TaskRegistry
     */
    private $taskRegistry;

    /**
     * @var LoggerInterface|MockObject
     */
    private $logger;

    /**
     * @var MockObject|TaskLocker
     */
    private $locker;

    public function testNextContext(): void
    {
        $taskMeta = $this->getMockBuilder(CronTaskMeta::class)->disableOriginalConstructor()->getMock();
        $task     = $this->getMockBuilder(CronTaskInterface::class)->disableOriginalConstructor()->getMock();
        $repo     = $this->getMockBuilder(EntityRepository::class)->disableOriginalConstructor()->getMock();
        $this->taskRegistry->expects(static::any())
            ->method('getRepository')
            ->willReturn($repo);

        $repo->expects(static::atLeast(1))
            ->method('findBy')
            ->willReturnOnConsecutiveCalls([], [$taskMeta]);

        // test no tasks found $repo call #1
        $context = $this->instance->nextContext();
        static::assertNull($context);

        $serviceId = 'App\\Task';

        $taskMeta->expects(static::once())
            ->method('shouldRun')
            ->willReturn(true);

        $taskMeta->expects(static::any())
            ->method('getServiceId')
            ->willReturn($serviceId);

        $this->logger->expects(static::once())
            ->method('debug')
            ->withConsecutive(['Locked task ' . $serviceId]);

        $this->taskRegistry->expects(static::once())
            ->method('findTask')
            ->withConsecutive([$serviceId])
            ->willReturn($task);

        $this->locker->expects(static::once())
            ->method('lockTask')
            ->willReturn(true);

        // test available taskMeta $repo call #2
        $context = $this->instance->nextContext();
        static::assertNotNull($context);
    }

    public function testNextContextCantRun(): void
    {
        $taskMeta = $this->getMockBuilder(CronTaskMeta::class)->disableOriginalConstructor()->getMock();
        $task     = $this->getMockBuilder(CronTaskInterface::class)->disableOriginalConstructor()->getMock();
        $repo     = $this->getMockBuilder(EntityRepository::class)->disableOriginalConstructor()->getMock();
        $this->taskRegistry->expects(static::any())
            ->method('getRepository')
            ->willReturn($repo);

        $repo->expects(static::atLeast(1))
            ->method('findBy')
            ->willReturn([$taskMeta]);

        $serviceId = 'App\\Task';

        $taskMeta->expects(static::atLeast(1))
            ->method('shouldRun')
            ->willReturnOnConsecutiveCalls(false, true, true);

        $taskMeta->expects(static::any())
            ->method('getServiceId')
            ->willReturn($serviceId);

        $this->taskRegistry->expects(static::atLeast(1))
            ->method('findTask')
            ->willReturnOnConsecutiveCalls(null, $task);

        $this->locker->expects(static::once())
            ->method('lockTask')
            ->willReturn(false);

        $this->logger->expects(static::atLeast(1))
            ->method('debug')
            ->withConsecutive(
                [$serviceId . ' does not need to run'],
                ["Task {$serviceId} got locked by another process"]
            );

        $this->logger->expects(static::once())
            ->method('warning')
            ->withConsecutive(["Task {$serviceId} not found. Meta out of sync?"]);

        // task should not run
        $context = $this->instance->nextContext();
        static::assertNull($context);

        // no task class
        $context = $this->instance->nextContext();
        static::assertNull($context);

        // task locked by another process
        $context = $this->instance->nextContext();
        static::assertNull($context);
    }

    /**
     * @dataProvider completeData
     *
     * @param mixed[] $unlockParams
     */
    public function testComplete(ScheduleContext $context, ?string $message, $unlockParams): void
    {
        $this->locker->expects(static::once())
            ->method('unlockTask')
            ->with(...$unlockParams);

        $this->instance->completed($context, $message);
    }

    /**
     * @return Generator<mixed>
     */
    public function completeData(): Generator
    {
        $taskMeta = $this->getMockBuilder(CronTaskMeta::class)->disableOriginalConstructor()->getMock();
        $task     = $this->getMockBuilder(CronTaskInterface::class)->disableOriginalConstructor()->getMock();

        $callback = static::callback(function ($value) {
            return $value instanceof DateTimeInterface;
        });

        $successContext = new ScheduleContext($task, $taskMeta);

        $failedContext = new ScheduleContext($task, $taskMeta);
        $failedContext->markFailed(new Exception('test error'));

        //last arg [CronTaskInfo $meta, array $messages, bool $disable, ?\DateTimeInterface $nextRun]
        yield 'success' => [
            $successContext,
            null,
            [$taskMeta, [], false, $callback],
        ];
        yield 'success with message' => [
            $successContext,
            'testMsg',
            [$taskMeta, ['testMsg'], false, $callback],
        ];
        yield 'failure exception' => [
            $failedContext,
            null,
            [$taskMeta, ['Exception: test error'], true, null],
        ];
    }

    protected function setUp(): void
    {
        $this->taskRegistry = $this->getMockBuilder(TaskRegistry::class)->disableOriginalConstructor()->getMock();
        $this->locker       = $this->getMockBuilder(TaskLocker::class)->disableOriginalConstructor()->getMock();
        $this->logger       = $this->getMockBuilder(LoggerInterface::class)->getMock();
        $this->instance     = new TaskScheduler($this->taskRegistry, $this->locker, $this->logger);
    }
}
