<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Command;

use Cyber\CronBundle\Entity\CronTaskMetaBase;
use Cyber\CronBundle\Manager\TaskRegistry;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class SyncMetadataCommand extends Command
{
    /**
     * @var TaskRegistry
     */
    private $taskRegistry;

    public function __construct(TaskRegistry $taskRegistry)
    {
        parent::__construct();
        $this->taskRegistry = $taskRegistry;
    }

    protected function configure(): void
    {
        $help = <<<'EOT'
Compares existing task classes with data in DB. Removes any tasks from DB that no longer exist and creates
any missing ones. Tasks will not execute unless their metadata has been synced to DB, so makes sure to run this
command whenever tasks are added or removed.

There is no harm in running this at any point, so you can just se this up to run during each deploy.
This should run AFTER any migrations (if used) or DB alterations.
EOT;

        $this->setDescription('Syncs task metadata to db.')
            ->setHelp($help);
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io    = new SymfonyStyle($input, $output);
        $tasks = $this->taskRegistry->getTaskClassMap();
        $repo  = $this->taskRegistry->getRepository();
        $em    = $this->taskRegistry->getObjectManager();

        $io->note(\sprintf('Application has %d task definitions', \count($tasks)));
        /** @var CronTaskMetaBase[] $taskMetas */
        $taskMetas = $repo->findAll();
        $io->note(\sprintf('Current metadata has %d definitions', \count($tasks)));
        $results = [];
        foreach ($taskMetas as $meta) {
            $class = $meta->getServiceId();
            if (isset($tasks[$class])) {
                // meta and task exist, remove task from array to avoid extra work
                unset($tasks[$class]);
                $results[] = [$class, 'UNCHANGED'];
                continue;
            }

            $results[] = [$class, 'REMOVED'];
            // meta exists, but task is gone, clean meta
            $em->remove($meta);
        }

        // whatever remains in $tasks are stuff that don't have meta yet
        foreach (\array_keys($tasks) as $class) {
            $meta = $this->taskRegistry->makeTaskMeta();
            $meta->setServiceId($class);
            $em->persist($meta);
            $results[] = [$class, 'ADDED'];
        }

        $io->table(['Task', 'Result'], $results);

        $em->flush();

        $io->success('Sync Completed');

        return 0;
    }
}
