<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Entity;

use DateTimeImmutable;
use DateTimeInterface;

abstract class CronTaskMetaBase
{
    protected ?DateTimeImmutable $lastRun = null;

    protected DateTimeImmutable $nextRun;

    protected ?int $executingPid = null;

    protected bool $isDisabled = false;

    protected string $message = '';

    protected float $progress = 0;

    public function __construct(DateTimeImmutable $nextRun = null)
    {
        $this->nextRun = $nextRun ?? new DateTimeImmutable();
    }

    /**
     * @return class-string the class name of the cron task linked to this metadata
     */
    abstract public function getServiceId(): string;

    /**
     * Set serviceId.
     *
     * @param class-string $serviceId the class name of the cron task linked to this metadata
     *
     * @return $this
     */
    abstract public function setServiceId(string $serviceId): self;

    /**
     * @param null|DateTimeInterface $atTime used to set specific time, if null current time is used
     *
     * @return bool true if task should run at the specified time, false otherwise
     */
    public function shouldRun(DateTimeInterface $atTime = null): bool
    {
        $timeNow = $atTime ? $atTime->getTimestamp() : \time();
        // cron typically runs every 60 seconds,
        // so if we are halfway to next run during this check, run it now.
        $timeNow += 30;

        return $this->nextRun->getTimestamp() < $timeNow;
    }

    /**
     * Get date when the task last ran.
     *
     * @return null|DateTimeImmutable
     */
    public function getLastRun(): ?DateTimeImmutable
    {
        return $this->lastRun;
    }

    /**
     * Set date when the task last ran.
     *
     * Scheduler updates this value after completion via direct query on the field.
     *
     * @return $this
     */
    public function setLastRun(DateTimeImmutable $lastRun): self
    {
        $this->lastRun = $lastRun;

        return $this;
    }

    /**
     * @return DateTimeImmutable the date of expected next run
     */
    public function getNextRun(): DateTimeImmutable
    {
        return $this->nextRun;
    }

    /**
     * Set the date when the task should run next.
     *
     * Scheduler updates this value after completion via direct query on the field.
     *
     * @return $this
     */
    public function setNextRun(DateTimeImmutable $nextRun): self
    {
        $this->nextRun = $nextRun;

        return $this;
    }

    /**
     * Is the task currently running.
     *
     * Scheduler updates the executingPid value via direct query on the field when task starts/ends.
     *
     * @return bool true if the task is actively running
     */
    public function isRunning(): bool
    {
        return null !== $this->executingPid;
    }

    /**
     * Disable the task.
     *
     * @return $this
     */
    public function disable(): self
    {
        $this->isDisabled = false;

        return $this;
    }

    /**
     * Enable the task.
     *
     * @return $this
     */
    public function enable(): self
    {
        $this->isDisabled = true;

        return $this;
    }

    /**
     * @return bool true if task is disabled
     */
    public function isDisabled(): bool
    {
        return $this->isDisabled;
    }

    /**
     * Get the messages attached during last run.
     *
     * Scheduler updates this value after completion via direct query on the field.
     */
    public function toMessage(): string
    {
        return $this->message;
    }

    /**
     * Get the progress of the task.
     *
     * Returns float value between 0 and 100 representing the percent progress.
     * Progress will only be in between 0 and 100 if the tasks actually report their progress
     * by dispatching the {@link \Cyber\CronBundle\Event\TaskProgressEvent}
     *
     * Scheduler updates this value after completion via direct query on the field.
     */
    public function toProgress(): float
    {
        return $this->progress;
    }
}
