# About
Simple task queue with Symfony and Cron daemon.

# Configuration

This bundle uses the following configuration structure. You should configure all options as needed by your project. You can find this configuration wth the command
```bash
$ php bin/console config:dump-reference cyber_cron
```

```yaml
cyber_cron:
    orm: # Required
        manager: doctrine.orm.entity_manager
        task_entity_class: ~ # Required
```

## Sample Configuration
First of all, you need to create an entity to store the information about tasks. Ex. */src/Entity/Cron.php*

Ultimately your entity must extends `Cyber\CronBundle\Entity\CronTaskMetaBase`, however the bundle does come
with a *Mapped Superclass* you can use, if our default mappings work for your project.

### Extending CronTaskMeta
The *Mapped Superclass* is `Cyber\CronBundle\Entity\CronTaskMeta`, if you extend that, you only need to define id field.
```php
<?php
namespace App\Entity;

use Cyber\CronBundle\Entity\CronTaskMeta;
use Doctrine\ORM\Mapping as ORM;

#[ORM\Table(name: "cron_task_meta")]
#[ORM\Entity]
class Cron extends CronTaskMeta
{
    #[ORM\Column]
    #[ORM\Id]
    #[ORM\GeneratedValue]
    protected ?int $id;
}
``` 

### Extending CronTaskMetaBase

For a more customized mapping you can extend `Cyber\CronBundle\Entity\CronTaskMetaBase`. You must provide mapping for
all protected fields for scheduler to work.

```php
<?php
namespace App\Entity;

use Cyber\CronBundle\Entity\CronTaskMetaBase;
use Doctrine\ORM\Mapping as ORM;

#[ORM\Table(name: "cron_task_meta")]
#[ORM\Entity]
class Cron extends CronTaskMetaBase
{
    #[ORM\Column]
    #[ORM\Id]
    #[ORM\GeneratedValue]
    protected ?int $id;
    
    #[ORM\Column(type: 'datetimetz_immutable', nullable: true)]
    protected ?DateTimeImmutable $lastRun = null;

    #[ORM\Column(type: 'datetimetz_immutable')]
    protected DateTimeImmutable $nextRun;

    #[ORM\Column(type: 'integer', nullable: true)]
    protected ?int $executingPid = null;
    
    // other rest of the protect fields from base class.
}
``` 

### Migration
Once your entity is created, if you are using doctrine migrations you should create a migration for your entity:
```bash
$ php bin\console make:migration
$ php bin\console doctrine:migrations:migrate
```

### Add Config Values
Then you need to configure the bundle via your project config files:
```yml
cyber_cron:
    orm:
        task_entity_class: 'App\Entity\Cron'
```

# Creating Tasks
> Below steps assume you have auto-configuration enabled. If not you will need to tag all your task services with tag
> `cyber.cron`

You have 2 options in creating tasks:
1. Create any arbitrary service class which implements `CronTaskInterface`
2. Make any console command implement `CronCommandInterface` to turn it into Cron task managed by this bundle

## Implementing CronTaskInterface

```php
<?php
// src/Services/CronTasker.php
namespace App\Services;

use Cyber\CronBundle\Component\CronTaskInterface;
use DateTimeImmutable;

class MyCronTask implements CronTaskInterface
{
    public function execute(): ?bool
    {
        // Do the work of the task here
        print_r("This is my first task.");
        // return FALSE or throw exception on failure
    }
    
    public function getNextExecutionTime(DateTimeImmutable $currentStartTime): DateTimeImmutable
    {
        // run task every hour
        return $currentStartTime->modify('+1 hour');
    }
}
```

## Implementing CronCommandInterface
```php
<?php
namespace App\Command;

use Cyber\CronBundle\Component\CronCommandInterface;
use DateTimeImmutable;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;

#[AsCommand('my:cron:test')]
class MyCronCommand extends Command implements CronCommandInterface
{
    /**
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $output->writeln('Command executed');

        return 0;
    }

    public function getNextExecutionTime(DateTimeImmutable $currentStartTime): DateTimeImmutable
    {
        return $currentStartTime->modify('1 minute');
    }
}
```

## Task Metadata
Scheduler depends on the metadata entity to run your tasks. Keep metadata up to date by executing the sync command
whenever you add/remove new tasks: 
```bash
$ php bin/console cyber:cron:sync-metadata
```

## Teating Cron Task
The debug command provided by this bundle will serve as both validating that your task metadata exist, as well as
perform a test executiong of your task. Simply pass your cron task FQCN as an argument:

```bash
$ php bin/console debug:cyber:cron App\Your\Task\Class
```

## Setup Crontab
As a final step you need to setup the crontab to execute the scheduler. Scheduler should be setup to run every minute:

```bash
* * * * * php bin/console cyber:cron:run > /var/log/cyber-cron.log
```
> The above is extremely condensed example. On your production systems you may need to provide full paths to executables
> and configure proper working directory. Also keep in mind cron should execute under user that runs the web server
> behind your application, otherwise cache permission conflicts may occur. 

### Crontab Schedule
You may choose to configure the scheduler to run as infrequently as your most frequent task. 
For example, if all your tasks are set to run once an hour, can change the definition to `0 * * * *`. This would save
on db query which will now run once an hour instead of once per minute.

# Events
CronBundle generates 3 types of events when the task is executed.
* Before Execution
* After successful completion
* After unsuccessful completion

You need to create a listener file (*listener.php) to subscribe to these events:
```php
class YourListener implements EventSubscriberInterface
{
    public static function getSubscribedEvents(): array
    {
        return [
            CyberCronEvents::TASK_START    => 'taskStart',
            CyberCronEvents::TASK_FINISHED => 'taskFinished',
            CyberCronEvents::TASK_FAILURE  => 'taskFailure',
        ];
    }

    /**
     * @param CronTaskInfoEvent $event
     */
    public function taskStart(CronTaskInfoEvent $event): void
    {
    }

    /**
     * @param CronTaskInfoEvent $event
     */
    public function taskFinished(CronTaskInfoEvent $event): void
    {
    }

    /**
     * @param CronTaskInfoEvent $event
     */
    public function taskFailure(CronTaskInfoEvent $event): void
    {
    }
}
```

And register it in the configuration file (*.yml):

```yaml
services:
    path\to\file\YourListener:
        tags:
            - { name: kernel.event_subscriber }
```

# Multi Server Environment
This bundle leverages DB for locking purposes. So if 2 servers try to run same task at the same time only one will
succeed. Which means that you can safely run this on any number of servers simultaneously and be sure that tasks would
not get repeated unnecessarily. 

If you do have very large number of servers we recommend adding a `RANDOM_DELAY=15` to your cron file, so that 
cron tasks start with a random delay of up to 15 minutes on each of the servers. This will somewhat mitigate race 
conditions where all servers will attempt to execute the same task on the same second of the day. 
