<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Manager;

use Closure;
use Cyber\CronBundle\Component\CronTaskBaseInterface;
use Cyber\CronBundle\Entity\CronTaskMetaBase;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\EntityRepository;
use Doctrine\ORM\Mapping\ClassMetadata;
use LogicException;

class TaskRegistry
{
    /** @var Closure[] */
    private $taskClosures;

    /** @var null|array<class-string<CronTaskBaseInterface>, CronTaskBaseInterface> */
    private $taskClassMap;

    /**
     * @var EntityManagerInterface
     */
    private $objectManager;

    /**
     * @var class-string<CronTaskMetaBase>
     */
    private $entityClass;

    /**
     * @var null|EntityRepository<CronTaskMetaBase>
     */
    private $repository;

    /**
     * @param class-string<CronTaskMetaBase> $entityClass
     */
    public function __construct(EntityManagerInterface $manager, string $entityClass)
    {
        $this->objectManager = $manager;
        $this->entityClass   = $entityClass;

        $this->taskClosures = [];
    }

    /**
     * @param Closure $task
     */
    public function addScheduledTaskClosure(Closure $task): void
    {
        $this->taskClosures[] = $task;
    }

    /**
     * @return EntityRepository<CronTaskMetaBase>
     */
    public function getRepository(): EntityRepository
    {
        if ($this->repository) {
            return $this->repository;
        }

        $repo = $this->objectManager->getRepository($this->entityClass);
        if (!$repo instanceof EntityRepository) {
            throw new LogicException('Only EntityRepository supported');
        }

        return $this->repository = $repo;
    }

    /**
     * @param class-string $taskClass
     *
     * @return null|CronTaskBaseInterface
     */
    public function findTask(string $taskClass): ?CronTaskBaseInterface
    {
        $map = $this->getTaskClassMap();

        return $map[$taskClass] ?? null;
    }

    /**
     * @return array<class-string<CronTaskBaseInterface>, CronTaskBaseInterface>
     */
    public function getTaskClassMap(): array
    {
        if (null !== $this->taskClassMap) {
            return $this->taskClassMap;
        }

        $this->taskClassMap = [];

        foreach ($this->taskClosures as $closure) {
            /** @var CronTaskBaseInterface $task */
            $task = $closure();

            $this->taskClassMap[\get_class($task)] = $task;
        }

        return $this->taskClassMap;
    }

    public function getObjectManager(): EntityManagerInterface
    {
        return $this->objectManager;
    }

    public function makeTaskMeta(): CronTaskMetaBase
    {
        return new $this->entityClass();
    }

    /**
     * @return ClassMetadata<CronTaskMetaBase>
     *
     * @noinspection PhpUnnecessaryLocalVariableInspection
     * @noinspection PhpRedundantVariableDocTypeInspection
     */
    public function getEntityMetadata(): ClassMetadata
    {
        /** @var ClassMetadata<CronTaskMetaBase> $meta stan does not infer properly */
        $meta = $this->objectManager->getClassMetadata($this->entityClass);

        return $meta;
    }
}
