> Below steps assume you have auto-configuration enabled. If not you will need to tag all your task services with tag
> `cyber.cron`

You have 2 options in creating tasks:

1. Create any arbitrary service class which implements `CronTaskInterface`
2. Make any console command implement `CronCommandInterface` to turn it into Cron task managed by this bundle

In both cases you will have an `execute()` method to do the work (albeit with slightly diffe signature), and
a `getNextExecutionTime()` method for deciding next execution time.

## Implementing CronTaskInterface

You can turn any service into a cron task by simply implementing `CronTaskInterface`

```php
<?php
// src/Services/CronTasker.php
namespace App\Services;

use Cyber\CronBundle\Component\CronTaskInterface;
use DateTimeImmutable;

class MyCronTask implements CronTaskInterface
{
    public function execute(): ?bool
    {
        // Do the work of the task here
        print_r("This is my first task.");
        // return FALSE or throw exception on failure
    }
    
    public function getNextExecutionTime(DateTimeImmutable $currentStartTime): DateTimeImmutable
    {
        // run task every hour
        return $currentStartTime->modify('+1 hour');
    }
}
```

## Implementing CronCommandInterface

```php
<?php
namespace App\Command;

use Cyber\CronBundle\Component\CronCommandInterface;
use DateTimeImmutable;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;

#[AsCommand('my:cron:test')]
class MyCronCommand extends Command implements CronCommandInterface
{
    /**
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $output->writeln('Command executed');

        return 0;
    }

    public function getNextExecutionTime(DateTimeImmutable $currentStartTime): DateTimeImmutable
    {
        return $currentStartTime->modify('1 minute');
    }
    
    public function getCronArguments(): array
    {
        // use this to set arguments to be used when executed by cron scheduler when defaults are not suitable.
        return [];
    }
}
```

## Next Execution Time

The `getNextExecutionTime()` must return a date object with the time when next iteration of this task should be
executed.

As an argument it is passed the time when the current execution was started. It is there for cases when subsequent
execution need to be relative to the previous one. 
