<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\CronBundle\Command;

use Cyber\CronBundle\Manager\TaskRegistry;
use Cyber\CronBundle\Manager\TaskRunner;
use Exception;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Throwable;

#[AsCommand(name: 'debug:cyber:cron', description: 'Run cron task manually.')]
class DebugCronCommand extends Command
{
    private $taskRegistry;

    public function __construct(TaskRegistry $taskTaskRegistry)
    {
        parent::__construct();
        $this->taskRegistry = $taskTaskRegistry;
    }

    protected function configure(): void
    {
        $this->addArgument('task', InputArgument::REQUIRED, 'Execute specific task regardless of the schedule.');
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @throws Exception
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io        = new SymfonyStyle($input, $output);
        $taskClass = $input->getArgument('task');

        if (!\class_exists($taskClass)) {
            $io->error(\sprintf('Class "%s" does not exist.', $taskClass));

            return 1;
        }

        $task = $this->taskRegistry->findTask($taskClass);

        if (null === $task) {
            $io->error(\sprintf('Task "%s" was not found.', $taskClass));

            return 1;
        }

        $io->note(\sprintf('Executing task "%s"', $taskClass));

        $runner = new TaskRunner();

        try {
            $message = $runner->run($task);

            $io->success('Finished Task');
            if ($message && $output->isVerbose()) {
                $io->block($message, 'OUTPUT');
            }

            return 0;
        } catch (Throwable $t) {
            $io->error(\sprintf('Task returned "%s"', $t->getMessage()));

            return 1;
        }
    }
}
