<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Command;

use Cyber\CronBundle\Manager\TaskRegistry;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

#[AsCommand(name: 'cyber:cron:list', description: 'Lists all discovered tasks.')]
class ListCommand extends Command
{
    /**
     * @var TaskRegistry
     */
    private $taskRegistry;

    public function __construct(TaskRegistry $taskRegistry)
    {
        parent::__construct();
        $this->taskRegistry = $taskRegistry;
    }

    protected function configure(): void
    {
        $help = <<<'EOT'
You can use this command to check on the status of metadata sync as well as each of the tasks execution status.
EOT;

        $this->setHelp($help);
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io    = new SymfonyStyle($input, $output);
        $tasks = $this->taskRegistry->getTaskClassMap();
        $repo  = $this->taskRegistry->getRepository();

        $taskMetas = $repo->findAll();
        $results   = [];

        foreach ($taskMetas as $meta) {
            $class  = $meta->getServiceId();
            $status = 'Present (Task Removed)';
            if (isset($tasks[$class])) {
                // meta and task exist, remove task from array to avoid extra work
                unset($tasks[$class]);
                $status = 'Present';
            }

            $results[] = [
                $class,
                $status,
                $meta->getLastRun()?->format('m/d/y H:i:s') ?? 'n/a',
                $meta->getNextRun()->format('m/d/y H:i:s T'),
                $meta->isRunning() ? 'Y' : 'N',
                $meta->isDisabled() ? 'Y' : 'N',
            ];
        }

        // whatever remains in $tasks are stuff that don't have meta yet
        foreach (\array_keys($tasks) as $class) {
            $results[] = [$class, 'Missing (New Task)', 'n/a', 'n/a', 'n/a', 'n/a'];
        }

        $io->table(['Task', 'Metadata Status', 'Last Run', 'Next Run', 'Running', 'Disabled'], $results);

        return 0;
    }
}
