<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Tests\Cyber\CronBundle\Command;

use Cyber\CronBundle\Command\CronCommand;
use Cyber\CronBundle\Component\CronTaskInterface;
use Cyber\CronBundle\Entity\CronTaskMeta;
use Cyber\CronBundle\Event\TaskFinishedEvent;
use Cyber\CronBundle\Event\TaskStartEvent;
use Cyber\CronBundle\Manager\ScheduleContext;
use Cyber\CronBundle\Manager\TaskScheduler;
use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Tester\CommandTester;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * @covers \Cyber\CronBundle\Command\CronCommand
 *
 * @internal
 */
class CronCommandTest extends TestCase
{
    /**
     * @var MockObject|TaskScheduler
     */
    private $scheduler;

    /**
     * @var EventDispatcherInterface|MockObject
     */
    private $dispatcher;

    /**
     * @var CronCommand
     */
    private $command;

    public function testNoTasksCommand(): void
    {
        $commandTester = new CommandTester($this->command);
        $commandTester->execute([]);

        $output = $commandTester->getDisplay();
        static::assertStringContainsString('No pending tasks.', $output);
    }

    public function testWithTask(): void
    {
        $task    = $this->getMockBuilder(CronTaskInterface::class)->getMock();
        $command = new MockCronCommand();
        $meta    = $this->getMockBuilder(CronTaskMeta::class)->getMock();

        $context     = new ScheduleContext($task, $meta);
        $commandCtxt = new ScheduleContext($command, $meta);
        $this->scheduler->expects(static::atLeast(1))
            ->method('nextContext')
            ->willReturnOnConsecutiveCalls($context, $commandCtxt);

        $this->dispatcher->expects(static::atLeast(2))
            ->method('dispatch')
            ->withConsecutive([
                static::callback(function ($event) use ($context) {
                    self::assertInstanceOf(TaskStartEvent::class, $event);
                    self::assertSame($context, $event->toContext());

                    return true;
                }),
            ], [
                static::callback(function ($event) use ($context) {
                    self::assertInstanceOf(TaskFinishedEvent::class, $event);
                    self::assertSame($context, $event->toContext());

                    return true;
                }),
            ])->willReturnArgument(0);

        $commandTester = new CommandTester($this->command);
        $commandTester->execute([], ['verbosity' => OutputInterface::VERBOSITY_VERBOSE]);

        $output = $commandTester->getDisplay();
        static::assertStringContainsString('SUCCESS', $output);
        static::assertStringContainsString('Command executed', $output);
        static::assertStringNotContainsString('FAILURE', $output);
    }

    protected function setUp(): void
    {
        $this->scheduler  = $this->getMockBuilder(TaskScheduler::class)->disableOriginalConstructor()->getMock();
        $this->dispatcher = $this->getMockBuilder(EventDispatcherInterface::class)->getMock();
        $this->command    = new CronCommand($this->scheduler, $this->dispatcher);
    }
}
