<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Command;

use Cyber\CronBundle\Entity\CronTaskMetaBase;
use Cyber\CronBundle\Manager\TaskRegistry;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

#[AsCommand(name: 'cyber:cron:task', description: 'Update properties of single task.')]
class TaskCommand extends Command
{
    /**
     * @var TaskRegistry
     */
    private $taskRegistry;

    public function __construct(TaskRegistry $taskRegistry)
    {
        parent::__construct();
        $this->taskRegistry = $taskRegistry;
    }

    protected function configure(): void
    {
        $this->addArgument('task-class', InputArgument::REQUIRED, 'FQCN of the class to update')
            ->addOption('reset', 'r', InputOption::VALUE_NONE, 'Reset the running status and enable the task.')
            ->addOption('disable', 'd', InputOption::VALUE_NONE, 'Disable the task from future execution.');
        $help = <<<'EOT'
Use this to reset stuck tasks or disable tasks.
EOT;

        $this->setHelp($help);
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);

        $class   = $input->getArgument('task-class');
        $reset   = $input->getOption('reset');
        $disable = $input->getOption('disable');

        $taskMeta = $this->findMetaForClass($class);
        $em       = $this->taskRegistry->getObjectManager();

        if (!$taskMeta) {
            $io->error('Task "' . $class . '" not found, make sure the name is valid and metadata is synchronized');

            return Command::FAILURE;
        }

        $io->section('Current Task Details');
        $io->horizontalTable(
            ['Task', 'Last Run', 'Next Run', 'Running', 'Disabled'],
            [
                [
                    $class,
                    $taskMeta->getLastRun()?->format('m/d/y H:i:s T') ?? 'n/a',
                    $taskMeta->getNextRun()->format('m/d/y H:i:s T'),
                    $taskMeta->isRunning() ? 'Y' : 'N',
                    $taskMeta->isDisabled() ? 'Y' : 'N',
                ],
            ]
        );

        $printUpdate = false;

        if ($reset) {
            $taskMeta->reset();
            $printUpdate = true;
        }

        if ($disable) {
            $taskMeta->disable();
            $printUpdate = true;
        }

        $em->flush();

        if ($printUpdate) {
            $io->section('Updated Task Details');
            $io->horizontalTable(
                ['Task', 'Last Run', 'Next Run', 'Running', 'Disabled'],
                [
                    [
                        $class,
                        $taskMeta->getLastRun()?->format('m/d/y H:i:s T') ?? 'n/a',
                        $taskMeta->getNextRun()->format('m/d/y H:i:s T'),
                        $taskMeta->isRunning() ? 'Y' : 'N',
                        $taskMeta->isDisabled() ? 'Y' : 'N',
                    ],
                ]
            );
        }

        $io->success('Task "' . $class . '" updated successfully.');

        return Command::SUCCESS;
    }

    private function findMetaForClass(mixed $class): ?CronTaskMetaBase
    {
        $repo      = $this->taskRegistry->getRepository();
        $taskMetas = $repo->findAll(); // column could be arbitrary so we need to loop through all
        foreach ($taskMetas as $meta) {
            if ($meta->getServiceId() === $class) {
                return $meta;
            }
        }

        return null;
    }
}
