<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\CronBundle\Manager;

use Cyber\CronBundle\Entity\CronTaskMetaBase;
use Psr\Log\LoggerInterface;
use Psr\Log\NullLogger;

class TaskScheduler
{
    public function __construct(
        private readonly TaskRegistry $taskRegistry,
        private readonly TaskLocker $locker,
        private readonly LoggerInterface $logger = new NullLogger()
    ) {
    }

    /**
     * Finds and locks next task in the queue.
     *
     * The task must be returned to scheduler via completed() method to schedule next execution.
     *
     * @return ScheduleContext
     */
    public function nextContext(): ?ScheduleContext
    {
        $repo = $this->taskRegistry->getRepository();

        // must clear before each search otherwise we might get outdated data
        $this->taskRegistry->getObjectManager()->clear();

        /** @var CronTaskMetaBase[] $taskMetas */
        $taskMetas = $repo->findBy(
            ['isDisabled' => false, 'executingPid' => null],
            ['nextRun' => 'ASC'],
            10
        );
        foreach ($taskMetas as $taskMeta) {
            $id = $taskMeta->getServiceId();
            if (!$taskMeta->shouldRun()) {
                $this->logger->debug(\sprintf('%s does not need to run', $id));

                // since we are in ascending order once we hit one that shouldn't run the rest won't either
                return null;
            }
            $task = $this->taskRegistry->findTask($id);
            if (!$task) {
                $this->logger->warning(\sprintf('Task %s not found. Meta out of sync?', $id));
                continue;
            }

            if ($this->locker->lockTask($taskMeta)) {
                $this->logger->debug(\sprintf('Locked task %s', $id));

                return new ScheduleContext($task, $taskMeta);
            }

            $this->logger->debug(\sprintf('Task %s got locked by another process', $id));
        }

        return null;
    }

    public function completed(ScheduleContext $context, ?string $message): void
    {
        $task     = $context->getTask();
        $meta     = $context->getMeta();
        $messages = $message ? [$message] : [];
        $disable  = $context->failed();
        $nextRun  = null;

        if ($disable) {
            $messages[] = 'Exception: ' . $context->toFailureException()->getMessage();
        }

        if (!$disable) {
            $nextRun = $task->getNextExecutionTime($context->getStartTime());
        }

        $this->locker->unlockTask($meta, $messages, $disable, $nextRun);
    }
}
