<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\DeploymentBundle\Command;

use RuntimeException;
use SebastianBergmann\Environment\Runtime;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

/**
 * A wrapper class for application.
 *
 * Finds and runs a specific command via the application provided in a constructor.
 *
 * This is used to simplify testing of the "meta" commands (commands that call other commands on an application). Such
 * commands should use this invoker, so that during tests the application can be easily "mocked" by overriding "find"
 * and "run" functions. Trying to use a mocked application directly int he command is quite more complex as many more
 * functions, which get called from within command on the application object, need to be mocked.
 */
trait AppCommandInvoker
{
    /** @var OutputInterface */
    private $output;

    /** @var SymfonyStyle */
    private $invokerIo;

    /** @var string[] */
    private $proxyParams = [];

    /** @var string[] */
    private $extraArgs = [];

    public function initialize(InputInterface $input, OutputInterface $output): void
    {
        if (\is_callable('parent::initialize')) {
            parent::initialize($input, $output);
        }
        $this->output    = $output;
        $this->invokerIo = new SymfonyStyle($input, $output);

        foreach ($this->proxyParams as $param) {
            if ($input->hasOption($param)) {
                $value = $input->getOption($param);
                if (\is_array($value)) {
                    throw new RuntimeException('Cannot proxy array parameter: ' . $param);
                }

                $this->extraArgs[] = \sprintf('--%s=%s', $param, $value);
            }
        }

        if (!$input->isInteractive()) {
            //if current input is not interactive, the invoked commands should also be non-interactive
            $this->extraArgs[] = '-n';
        }

        $verbosity = $this->getVerbosity($output);
        if ($verbosity) {
            //if current output has verbosity set it for this output as well
            $this->extraArgs[] = $verbosity;
        }
    }

    /**
     * Finds the command name in the application and runs it using provided $input and $output as the command's input
     * and output.
     *
     * @param string               $commandName the name of the command
     * @param array<int|string>    $args        command arguments
     * @param null|OutputInterface $output      will redirect output to this interface instead of std
     *
     * @return int
     *
     * @SuppressWarnings(PHPMD.Superglobals)
     */
    public function invoke($commandName, array $args = [], OutputInterface $output = null): int
    {
        $descriptorspec = [
            0 => STDIN,  // stdin
            1 => $output ? ['pipe', 'w'] : \fopen('php://stdout', 'wb+'),  // stdout
            2 => \fopen('php://stderr', 'wb+'),  // stderr
        ];

        $binary = (new Runtime())->getBinary();
        if ('win' === \mb_strtolower(\mb_substr(PHP_OS, 0, 3))) {
            //weird problem on windows, must start with 2 double quotes (first one is added by escapeshellarg() above
            $binary = '"' . $binary;
        }
        $console = $_SERVER['SCRIPT_NAME'];

        $args = \array_merge($args, $this->extraArgs);

        $args = \array_map(function ($element) {
            return \escapeshellarg((string) $element);
        }, $args);

        $cmdLine = $binary . ' ' . \escapeshellarg($console) . ' ' . \escapeshellarg($commandName);
        if (!empty($args)) {
            $cmdLine .= ' ' . \implode(' ', $args);
            if ($this->output->isVerbose()) {
                $this->invokerIo->block($cmdLine, 'Command', 'fg=yellow', ' ! ');
            }
        }

        $process = proc_open($cmdLine, $descriptorspec, $pipes);
        if (false === $process) {
            throw new RuntimeException('Failed to start process for command: ' . $cmdLine);
        }

        if ($output) {
            $output->write(\stream_get_contents($pipes[1]) ?: '');
        }

        return proc_close($process);
    }

    /**
     * Sets which options from the original user input to pass on to all called commands.
     *
     * This must be called during command configuration {@link \Symfony\Component\Console\Command\Command::configure()}
     *
     * @param string[] $params
     *
     * @return $this
     */
    protected function setProxyOptions(array $params): self
    {
        $this->proxyParams = $params;

        return $this;
    }

    /**
     * @param OutputInterface $output
     *
     * @return false|string
     */
    private function getVerbosity(OutputInterface $output)
    {
        switch ($output->getVerbosity()) {
            case OutputInterface::VERBOSITY_DEBUG:
                return '-vvv';
            case OutputInterface::VERBOSITY_VERY_VERBOSE:
                return '-vv';
            case OutputInterface::VERBOSITY_VERBOSE:
                return '-v';
            case OutputInterface::VERBOSITY_QUIET:
                return '-q';
        }

        return false;
    }
}
