<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\DeploymentBundle;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Driver\ResultStatement;
use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\Migrations\Version\Version;
use PHPUnit\Framework\TestCase;

class MigrationHelpersTest extends TestCase
{
    /** @var MockMigrationHelper */
    private $helper;

    /**
     * @var \PHPUnit\Framework\MockObject\MockObject
     */
    private $mockVersion;

    /**
     * @var Connection|\PHPUnit\Framework\MockObject\MockObject
     */
    private $mockConnection;

    /**
     * @SuppressWarnings("PMD.UnusedLocalVariable")
     */
    protected function setUp(): void
    {
        $mockPlatform         = $this->getMockBuilder(AbstractPlatform::class)->getMock();
        $this->mockVersion    = $this->getMockBuilder(Version::class)->disableOriginalConstructor()->getMock();
        $this->mockConnection = $this->getMockBuilder(Connection::class)->disableOriginalConstructor()->getMock();

        $mockPlatform->method('quoteIdentifier')
            ->willReturnArgument(0);

        $this->helper = new MockMigrationHelper($mockPlatform, $this->mockVersion, $this->mockConnection);
    }

    public function testGetUuid4Sql(): void
    {
        $actual = $this->helper->getUuid4Sql();
        $this->assertEquals(4, \mb_substr_count($actual, '-'));
        $this->assertGreaterThan(0, \mb_strpos($actual, 'uuid4'));

        $actual = $this->helper->getUuid4Sql('somename', '_');
        $this->assertEquals(4, \mb_substr_count($actual, '_'));
        $this->assertGreaterThan(0, \mb_strpos($actual, 'somename'));
    }

    public function testGetUuid4BinSql(): void
    {
        $actual = $this->helper->getUuid4BinSql();
        $this->assertContains('UNHEX', $actual);
        $this->assertEquals(0, \mb_substr_count($actual, '-'));
        $this->assertGreaterThan(0, \mb_strpos($actual, 'uuid4'));

        $actual = $this->helper->getUuid4BinSql('otherName');
        $this->assertContains('UNHEX', $actual);
        $this->assertEquals(0, \mb_substr_count($actual, '-'));
        $this->assertGreaterThan(0, \mb_strpos($actual, 'otherName'));
    }

    public function testSomething(): void
    {
        $this->mockVersion
            ->expects($this::exactly(6))
            ->method('addSql')
            ->withConsecutive(
                ['SET foreign_key_checks=OFF'],
                ['ALTER TABLE src ADD CONSTRAINT FK FOREIGN KEY (col1) REFERENCES trg (col2) , ALGORITHM = INPLACE'],
                ['SET foreign_key_checks=ON'],
                ['SET foreign_key_checks=OFF'],
                ['ALTER TABLE src ADD CONSTRAINT FK FOREIGN KEY (col1) REFERENCES trg (col2) ON DELETE CASCADE, ALGORITHM = INPLACE'],
                ['SET foreign_key_checks=ON']
            );

        $this->helper->concurrentAddForeignKey('FK', 'src', 'trg', 'col1', 'col2');
        $this->helper->concurrentAddForeignKey('FK', 'src', 'trg', 'col1', 'col2', 'CASCADE');
    }

    public function testConcurrentAddIndex(): void
    {
        $this->mockVersion
            ->expects($this->exactly(2))
            ->method('addSql')
            ->withConsecutive(
                ['CREATE INDEX test_index ON alerts (alt_internal, alt_external) ALGORITHM = INPLACE'],
                ['CREATE UNIQUE INDEX test_index ON alerts (alt_internal, alt_external) ALGORITHM = INPLACE']
            );

        $this->helper->concurrentAddIndex('alerts', 'test_index', ['alt_internal', 'alt_external']);
        $this->helper->concurrentAddIndex('alerts', 'test_index', ['alt_internal', 'alt_external'], true);
    }

    public function testConcurrentRemoveIndex(): void
    {
        $this->mockVersion
            ->expects($this->once())
            ->method('addSql')
            ->with('DROP INDEX test_index ON alerts ALGORITHM = INPLACE');

        $this->helper->concurrentRemoveIndex('alerts', 'test_index');
    }

    public function testUpdatedInBatches(): void
    {
        $resultStm = $this->getMockBuilder(ResultStatement::class)->getMock();

        $this->mockConnection->expects($this->once())
            ->method('executeQuery')
            ->willReturn($resultStm);

        $resultStm->expects($this->once())
            ->method('fetch')
            ->willReturn(['ct' => 5]);

        $this->mockVersion
            ->expects($this->exactly(4))
            ->method('addSql')
            ->withConsecutive(
                ['CREATE INDEX tmp_idx_1999dbe27c13 ON alerts (message) ALGORITHM = INPLACE'],
                ['UPDATE alerts SET message = \'test message\' WHERE message != \'test message\' OR \'test message\' IS NULL AND message IS NOT NULL LIMIT 1000'],
                ['UPDATE alerts SET message = \'test message\' WHERE message != \'test message\' OR \'test message\' IS NULL AND message IS NOT NULL'],
                ['DROP INDEX tmp_idx_1999dbe27c13 ON alerts ALGORITHM = INPLACE']
            );

        $this->helper->updateColumnInBatches('alerts', 'message', '\'test message\'');
    }
}
