<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\DeploymentBundle\Command;

use Cyber\DeploymentBundle\Migrations\MergeMigrationException;
use Cyber\DeploymentBundle\Migrations\MigrationMerger;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\DependencyInjection\ContainerAwareInterface;
use Symfony\Component\DependencyInjection\ContainerAwareTrait;
use Symfony\Component\Filesystem\Filesystem;

class MergeMigrationsCommand extends Command implements ContainerAwareInterface
{
    use ContainerAwareTrait;

    /** @var SymfonyStyle */
    private $io;

    /** @var string */
    private $sourceDir;

    /** @var string */
    private $destDir;

    /** @var string[] */
    private $categories;

    /** @var Filesystem */
    private $filesystem;

    /**
     * {@inheritdoc}
     */
    public function __construct(Filesystem $filesystem)
    {
        parent::__construct();
        $this->filesystem = $filesystem;
    }

    protected function configure(): void
    {
        $this
            ->setName('cyber:deploy:merge-migrations')
            ->setDescription('Merges pending dev migrations into single prod migration');
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $this->io = new SymfonyStyle($input, $output);

        $container        = $this->container;
        $this->sourceDir  = $container->getParameter('cyber.deployment.migrations.dev');
        $this->destDir    = $container->getParameter('cyber.deployment.migrations.prod');
        $this->categories = $container->getParameter('cyber.deployment.migrations.categories');

        if (empty($this->categories)) {
            // set to '.' if no categories configured to use dev and prod directories themselves
            $this->categories = ['.'];
        }

        $this->filesystem->mkdir($this->destDir);

        foreach ($this->categories as $category) {
            $path = $this->destDir . DIRECTORY_SEPARATOR . $category;
            $this->filesystem->mkdir($path);
        }

        if (null === $this->sourceDir || null === $this->destDir) {
            $this->io->error('Invalid configuration. Make sure you have provided configuration for cyber_deployment.migrations');

            return 1;
        }

        $versions  = [];
        $namespace = $this->container->getParameter('doctrine_migrations.namespace');

        $merger = new MigrationMerger($this->io, $namespace, $this->sourceDir, $this->destDir);

        foreach ($this->categories as $category) {
            $ver = $this->processCategory($merger, $category);
            if (null !== $ver) {
                $versions[] = $ver;
                \sleep(1); //sleep for 1 second to ensure unique version number if multiple categories are present
            }
        }

        if (empty($versions)) {
            $this->io->note('No new migrations');

            return 0;
        }

        $this->io->success('Migration(s) have been generated in:');
        foreach ($versions as $versionPath) {
            $this->io->success('  - ' . $versionPath);
        }
        $this->io->comment('!!!ADD TO GIT!!!');

        return 0;
    }

    private function processCategory(MigrationMerger $merger, string $category): ?string
    {
        try {
            $ver = $merger->merge($category);
            if (null !== $ver) {
                return $ver;
            }
            $this->io->note('No new versions in ' . $category);
        } catch (MergeMigrationException $e) {
            foreach ($e->toErrors()->listErrors() as $error) {
                $this->io->error($error);
            }
        }

        return null;
    }
}
