<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\DeploymentBundle\Command;

use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

class NoDowntimeUpdateCommand extends AbstractNoDowntimeCommand
{
    protected function configure(): void
    {
        $this
            ->setName('cyber:no-downtime:update')
            ->setDescription('Helps you manage your post deployment migrations');
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     * @SuppressWarnings(PHPMD.UnusedLocalVariable)
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     * @SuppressWarnings(PHPMD.LongVariable)
     * @SuppressWarnings(PHPMD.ElseExpression)
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);

        $migrationSequence      = $this->sequence['post_deploy_migrations'];
        $sequenceVersions       = \array_keys($migrationSequence);
        $preDeployVersions      = $this->readVersions($this->container->getParameter('cyber.deployment.no_downtime.pre'));
        $postDeployVersions     = $this->readVersions($this->container->getParameter('cyber.deployment.no_downtime.post'));
        $lastSequenceVersion    = (int) \end($sequenceVersions);
        $lastPreDeploy          = (int) \end($preDeployVersions);
        $maxSequencedPostDeploy = 0;

        foreach ($migrationSequence as $preVersion => $postVersions) {
            foreach ($postVersions as $postVersion) {
                $maxSequencedPostDeploy = \max($maxSequencedPostDeploy, $postVersion);
                if (isset($postDeployVersions[$postVersion])) {
                    unset($postDeployVersions[$postVersion]);
                } else {
                    $io->warning('A post deploy version ' . $postVersion . ' is specified in sequence file but does not exist.');
                }
            }
        }

        if ($lastPreDeploy < $lastSequenceVersion) {
            $io->error(\sprintf(
                'The latest available migration version (%s) is lower than latest sequence version (%s). ' .
                'This could be caused by a deleted migration. Aborting to avoid data corruption',
                $lastPreDeploy,
                $lastSequenceVersion
            ));

            return 1;
        }

        if ($lastPreDeploy < $maxSequencedPostDeploy) {
            $io->error(
                \sprintf(
                    'The latest available migration version (%s) is lower than the latest sequence version (%s). ' .
                    'This could be caused by out or order commits. Rename migrations and try again. ' .
                    'Aborting to avoid data corruption.',
                    $lastPreDeploy,
                    $maxSequencedPostDeploy
                )
            );

            return 1;
        }

        $earlyVersions = [];

        foreach ($postDeployVersions as $versionName) {
            if ($versionName < $lastSequenceVersion) {
                $io->warning('A post-deploy version ' . $versionName . ' is earlier than latest sequence version ' . $lastSequenceVersion . '; this could indicate migration desync');
            }

            if ($versionName < $lastPreDeploy) {
                $earlyVersions[] = $versionName;
            }
        }

        if (!empty($earlyVersions)) {
            $io->error(\sprintf('The following post-deploy migrations have versions lower than latest pre-deploy (%s). ' .
                'Please re-name you post-deploy migrations to be of higher version than pre-deploy', $lastPreDeploy));
            $io->listing($earlyVersions);

            return 1;
        }

        if (empty($postDeployVersions)) {
            $io->note('No new post-deploy migrations found.');
            if ($lastPreDeploy !== $lastSequenceVersion) {
                $io->note('Updating sequence with pre-deploy version.');
                $this->commitMigrations($lastPreDeploy, []);
            }

            return 0;
        }

        $io->section('New Post-Deploy Migrations Found');

        if ($lastPreDeploy === $lastSequenceVersion) {
            $io->text('The following post-deploy migrations will be appended to sequence at version ' . $lastPreDeploy . ':');
            $io->listing($postDeployVersions);
            if (!$io->confirm('Do you wish to proceed?')) {
                return 0;
            }
        } else {
            $io->text('The following post-deploy migrations will be added to sequence at version ' . $lastPreDeploy . ':');
            $io->listing($postDeployVersions);
            if (!$io->confirm('Do you wish to proceed?')) {
                return 0;
            }
        }

        $this->commitMigrations($lastPreDeploy, \array_values($postDeployVersions));
        $io->success('Update sequence file with new migration versions.');

        return 0;
    }

    /**
     * @param int   $preDeployVersion
     * @param int[] $postDeployVersions
     */
    private function commitMigrations(int $preDeployVersion, $postDeployVersions): void
    {
        $sequence                    = $this->sequence['post_deploy_migrations'];
        $sequence[$preDeployVersion] = !isset($sequence[$preDeployVersion]) ? $postDeployVersions : \array_merge($sequence[$preDeployVersion], $postDeployVersions);

        $this->sequence['post_deploy_migrations'] = $sequence;
        $this->writeSequence();
    }
}
