<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\DeploymentBundle\Command;

use PHPUnit\Framework\MockObject\MockObject;
use PHPUnit\Framework\TestCase;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\BufferedOutput;

class ApplicationCommandInvokerTest extends TestCase
{
    /** @var callable */
    public static $mockProc;

    /**
     * @dataProvider invokeDataProvider
     *
     * @param string            $commandName
     * @param array<int|string> $args
     * @param BufferedOutput    $output
     * @param mixed             $returnValue
     *
     * @throws \ReflectionException
     */
    public function testInvoke($commandName, $args, BufferedOutput $output, $returnValue): void
    {
        self::$mockProc = function ($cmd, array $pipes) use ($commandName, $returnValue) {
            self::assertEquals(1, \preg_match('#[\'"]' . \preg_quote($commandName, '#') . '[\'"]#', $cmd));
            \fwrite($pipes[1], $commandName);
            \rewind($pipes[1]);

            return $returnValue;
        };

        /** @var MockAppCommandInvoker $invoker */
        $invoker = new MockAppCommandInvoker();

        /** @var InputInterface|MockObject $input */
        $input = $this->getMockBuilder(InputInterface::class)->getMock();

        $invoker->initialize($input, $output);

        $actualReturn = $invoker->invoke($commandName, $args, $output);

        $this->assertEquals($returnValue, $actualReturn);

        $data = $output->fetch();
        $this->assertEquals($commandName, $data);
    }

    /**
     * @return array<array<mixed>>
     */
    public function invokeDataProvider(): array
    {
        $output = new BufferedOutput();

        return [
            ['command:one', ['-n'], $output, 0],
            ['command:two', ['-n'], $output, 1],
            ['command:three', ['-n'], $output, 99],
        ];
    }
}

namespace Cyber\DeploymentBundle\Command;

use Tests\Cyber\DeploymentBundle\Command\ApplicationCommandInvokerTest;

/**
 * Override proc_open() that is used inside AppCommandInvoker.
 *
 * @param string       $cmd
 * @param array<mixed> $descriptorspec
 * @param array<mixed> $pipes
 *
 * @return bool|resource
 */
function proc_open($cmd, $descriptorspec, &$pipes)
{
    $pipes = [];
    foreach (\array_keys($descriptorspec) as $i) {
        $pipes[$i] = \fopen('php://memory', 'rb+');
    }
    $proc = ApplicationCommandInvokerTest::$mockProc;

    return $proc($cmd, $pipes);
}

/**
 * Override proc_close() that is used inside AppCommandInvoker.
 *
 * @param mixed $code
 *
 * @return mixed
 */
function proc_close($code)
{
    return $code;
}
