<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\DeploymentBundle\Command;

use Cyber\DeploymentBundle\Event\FixturesAppliedEvent;
use Cyber\DeploymentBundle\Event\PostDatabaseResetEvent;
use Cyber\DeploymentBundle\Event\PreDatabaseResetEvent;
use Doctrine\ORM\EntityManager;
use Doctrine\Persistence\ManagerRegistry;
use Generator;
use Psr\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\Console\Application;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\DependencyInjection\ContainerAwareInterface;
use Symfony\Component\DependencyInjection\ContainerAwareTrait;

/**
 * Class DevFixturesCommand.
 *
 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
 * @SuppressWarnings(PHPMD.NPathComplexity)
 */
class DevFixturesCommand extends Command implements ContainerAwareInterface
{
    use AppCommandInvoker;
    use ContainerAwareTrait;

    protected static $defaultName = 'cyber:deploy:fixtures';

    private EventDispatcherInterface $dispatcher;

    private ?ManagerRegistry $doctrine;

    public function __construct(EventDispatcherInterface $dispatcher, ?ManagerRegistry $doctrine)
    {
        $this->dispatcher = $dispatcher;
        $this->doctrine   = $doctrine;

        parent::__construct();
    }

    /**
     * command configuration.
     */
    protected function configure(): void
    {
        $this
            ->setDescription('Executes fixtures')
            ->addOption('dev', null, InputOption::VALUE_NONE, 'Apply dev fixtures')
            ->addOption('install', null, InputOption::VALUE_NONE, 'Apply fixtures for fresh installation');

        $this->setHelp(
            <<<EOT
Executes various types of fixtures.

If ran without any options, only the standard fixtures will be executed and database does not get dropped.

Run with --dev option to execute install and dev fixtures on top of standard ones. Database schema is dropped
and recreated based on current fixtures.

Run with --install to apply install and standard fixtures for new live deployments.
EOT
        );
        $this->setProxyOptions(['env']);
    }

    /**
     * @param InputInterface  $input
     * @param OutputInterface $output
     *
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);

        $container  = $this->container;
        $runDev     = $input->getOption('dev');
        $runInstall = $input->getOption('install');
        $app        = $this->getApplication();

        if (null === $app) {
            $io->error('Application instance is not set for the command!');

            return 1;
        }

        if (!$app->has('doctrine:fixtures:load')) {
            $io->error('It looks like you do not have fixtures enabled');
            $io->comment('composer require doctrine/doctrine-fixtures-bundle:^2.0');

            return 1;
        }

        if (!$this->doctrine) {
            $io->error('Doctrine is missing, make sure you have the bundle enabled and configured');

            return 1;
        }

        if ($runDev && $runInstall) {
            $io->error('--dev is intended for developer machines --install for new live deployments. Using both flags makes no sense. Aborting!.');

            return 1;
        }

        $connectionNames = $container->getParameter('cyber.deployment.fixture.connections') ?:
            [$this->doctrine->getDefaultConnectionName()];

        foreach ($connectionNames as $name) {
            // create database in case it does not exist
            if (0 !== $this->invoke(
                'doctrine:database:create',
                ['--if-not-exists', '--connection=' . $name],
                $output
            )) {
                $io->warning('Failed to create database. Maybe your platform does not support it. Assuming DB exists and continuing...');
            }
        }

        if ($runDev && !$this->handleDatabasePrep($input, $output, $app)) {
            return 1;
        }

        $unified = $this->container->getParameter('cyber.deployment.fixture.unified');

        $groups = \iterator_to_array($this->getUnifiedGroups($runInstall, $runDev));

        if ($unified) {
            if (!$this->handleUnifiedFixtures($io, $groups)) {
                $io->error('Failed to load fixtures');

                return 1;
            }
            $this->dispatcher->dispatch(new FixturesAppliedEvent($groups));

            return 0;
        }

        //append standard fixtures
        if (!$this->handleFixtures($io, 'standard', $runInstall || $runDev)) {
            $io->error('Failed to load standard fixtures');

            return 1;
        }

        // if we are running with --install or --dev flag also apply install fixtures
        if (($runInstall || $runDev) && !$this->handleFixtures($io, 'install', $runDev)) {
            $io->error('Failed to load install fixtures');

            return 1;
        }

        // load dev fixtures
        if ($runDev && !$this->handleFixtures($io, 'dev', false)) {
            $io->error('Failed to load dev fixtures');

            return 1;
        }
        $this->dispatcher->dispatch(new FixturesAppliedEvent($groups));

        return 0;
    }

    private function handleDatabasePrep(InputInterface $input, OutputInterface $output, Application $app): bool
    {
        $io        = new SymfonyStyle($input, $output);
        $container = $this->container;

        /** @var ManagerRegistry $doctrine */
        $doctrine = $container->get('doctrine');
        // [null] for default manager if none was configured
        $managerNames = $container->getParameter('cyber.deployment.fixture.managers') ?:
            [$doctrine->getDefaultManagerName()];

        $dbList = [];
        foreach ($managerNames as $name) {
            /** @var EntityManager $manager */
            $manager = $doctrine->getManager($name);

            $connection = $manager->getConnection();
            $params     = $connection->getParams();
            $dbList[]   = [$params['host'] ?? 'localhost', $connection->getDatabase()];
        }

        $io->note('The following databases will be reset');
        $io->table(['host', 'db'], $dbList);

        if ($input->isInteractive() &&
            !$io->confirm('Continue?', false)
        ) {
            $io->text('!!!User Aborted!!!');

            return false;
        }

        foreach ($managerNames as $name) {
            if (!$this->resetDatabase($app, $io, $name)) {
                $io->error('Failed to reset database: ' . $name);

                return false;
            }
        }

        return true;
    }

    private function handleFixtures(SymfonyStyle $io, string $type, bool $runningOthers): bool
    {
        if (!$this->container->getParameter('cyber.deployment.fixture.' . $type)) {
            if (!$runningOthers) { // no need for warning if we are running other fixtures
                $io->warning(\ucfirst($type) . ' fixtures are disabled.');
            }

            return true;
        }

        return 0 === $this->invoke('doctrine:fixtures:load', ['--append', '--group=' . $type], $io);
    }

    /**
     * @param SymfonyStyle $io
     * @param string[]     $groups
     *
     * @return bool
     */
    private function handleUnifiedFixtures(SymfonyStyle $io, array $groups): bool
    {
        $params = ['--append'];
        foreach ($groups as $group) {
            $params[] = '--group=' . $group;
        }

        return 0 === $this->invoke('doctrine:fixtures:load', $params, $io);
    }

    /**
     * @param Application  $app
     * @param SymfonyStyle $io
     *
     * @return bool
     *
     * @SuppressWarnings(PHPMD.ElseExpression)
     */
    private function resetDatabase(Application $app, SymfonyStyle $io, string $name): bool
    {
        /** @var PreDatabaseResetEvent $event */
        $event = $this->dispatcher->dispatch(new PreDatabaseResetEvent($name));
        if ($event->isAborted()) {
            $io->warning('Execution was aborted by event listener. Message was: ' . $event->getMessage());

            return false;
        }

        foreach ($event->getMessages() as $msg) {
            $io->comment($msg);
        }

        switch (true) {
            case $this->invoke('doctrine:schema:drop', ['--force', '--full-database', '--em=' . $name], $io):
            case $this->invoke('doctrine:schema:update', ['--complete', '--force', '--em=' . $name], $io):
                return false;
        }

        if ($app->has('doctrine:migrations:version')) {
            // ensure migration tables exit
            if ($this->invoke('doctrine:migrations:sync-metadata-storage', ['-n'], $io)) {
                return false;
            }

            // since we create schema from entities it means all migrations should be marked as executed
            if ($this->invoke('doctrine:migrations:version', ['--all', '--add', '-n'], $io)) {
                return false;
            }
        } else {
            $io->warning('You are missing migrations bundle. You should really use migrations');
        }

        /** @var PostDatabaseResetEvent $event */
        $event = $this->dispatcher->dispatch(new PostDatabaseResetEvent($name));
        if ($event->isAborted()) {
            $io->warning('Execution was aborted by event listener. Message was: ' . $event->getMessage());

            return false;
        }

        foreach ($event->getMessages() as $msg) {
            $io->comment($msg);
        }

        return true;
    }

    /**
     * @param bool $runInstall
     * @param bool $runDev
     *
     * @return Generator<string>
     */
    private function getUnifiedGroups(bool $runInstall, bool $runDev): Generator
    {
        yield 'standard';

        if ($runInstall || $runDev) {
            yield 'install';
        }

        if ($runDev) {
            yield 'dev';
        }
    }
}
