<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\DeploymentBundle\Command;

use Cyber\DeploymentBundle\Configuration;
use Cyber\DeploymentBundle\MigrationState;
use LogicException;
use RuntimeException;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;
use Symfony\Component\Yaml\Dumper;
use Symfony\Component\Yaml\Parser;
use UnexpectedValueException;

abstract class AbstractNoDowntimeCommand extends Command
{
    /** @var SymfonyStyle */
    protected $io;

    /** @var array<string, array<int, int[]>> */
    protected $sequence;

    public function __construct(protected Configuration $config)
    {
        parent::__construct();
    }

    protected function initialize(InputInterface $input, OutputInterface $output): void
    {
        $this->io = new SymfonyStyle($input, $output);

        $storagePath = $this->config->getPostDeployPath();

        if (null === $storagePath || !@\is_dir($storagePath)) {
            throw new RuntimeException('Storage path not configured or does not exist for post deploy migrations');
        }

        $this->sequence = $this->readCurrentSequence();
    }

    protected function getSequenceFilePath(): string
    {
        $storagePath = $this->config->getPostDeployPath();

        return $storagePath . DIRECTORY_SEPARATOR . 'sequence.yml';
    }

    protected function writeSequence(): void
    {
        $dumper = new Dumper();
        $file   = $this->getSequenceFilePath();
        \file_put_contents($file, $dumper->dump($this->sequence, 4));
    }

    /**
     * @return array<string, array<int, int[]>>
     */
    private function readCurrentSequence(): array
    {
        $sequenceFile = $this->getSequenceFilePath();

        if (!\is_file($sequenceFile)) {
            $this->io->note('Sequence file not found generating default one in: ' . $sequenceFile);
            $defaultConfig = ['post_deploy_migrations' => []];
            $dumper        = new Dumper();
            \file_put_contents($sequenceFile, $dumper->dump($defaultConfig, 4));

            return $defaultConfig;
        }

        $sequenceData = \file_get_contents($sequenceFile);
        if (false === $sequenceData) {
            throw new RuntimeException('Failed to read sequence file: ' . $sequenceFile);
        }

        $parser = new Parser();
        /** @var array<string, array<int, int[]>> $data */
        $data = $parser->parse($sequenceData);

        return $data;
    }

    /**
     * @param string $path
     *
     * @return int[]
     */
    protected function readVersions(string $path): array
    {
        $versions = \scandir($path, SCANDIR_SORT_ASCENDING);
        if (false === $versions) {
            throw new UnexpectedValueException('Failed to read version from path: ' . $path);
        }

        $versionNames = [];

        foreach ($versions as $version) {
            if (\preg_match('#^Version(\d*)\.php$#', $version, $matches)) {
                $versionNames[$matches[1]] = (int) $matches[1];
            }
        }

        return $versionNames;
    }

    protected function loadMetadata(): MigrationState
    {
        $prePath  = $this->config->getPreDeployPath();
        $postPath = $this->config->getPostDeployPath();

        if(!$prePath || !$postPath) {
            throw new LogicException('Both pre and post deploy paths must be configured.');
        }

        $migrationSequence      = $this->sequence['post_deploy_migrations'];
        $preDeployVersions      = $this->readVersions($prePath);
        $postDeployVersions     = $this->readVersions($postPath);

        return new MigrationState($migrationSequence, $preDeployVersions, $postDeployVersions);
    }
}
