<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\DeploymentBundle\Command;

use Exception;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\BufferedOutput;
use Symfony\Component\Console\Output\OutputInterface;

#[AsCommand('cyber:no-downtime:execute')]
class NoDowntimeExecuteCommand extends AbstractNoDowntimeCommand
{
    use AppCommandInvoker;

    protected function configure(): void
    {
        $this->setDescription('Executes post deployment migrations')
            ->setProxyOptions(['env'])
            ->addOption(
                'dry-run',
                null,
                InputOption::VALUE_NONE,
                'Show queries but do not actually execute migrations.'
            )
            ->addOption('pre', null, InputOption::VALUE_NONE, 'Execute pre-deploy migrations')
            ->addOption('post', null, InputOption::VALUE_NONE, 'Execute post-deploy migrations')
            ->addOption(
                'force',
                null,
                InputOption::VALUE_NONE,
                'Force execution of all needed migrations when migrating multiple sequence versions at once.'
            );
    }

    /**
     * @SuppressWarnings(PHPMD.CyclomaticComplexity)
     * @SuppressWarnings(PHPMD.NPathComplexity)
     *
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        /** @var string $env */
        $env   = $input->getOption('env');
        $pre   = $input->getOption('pre');
        $post  = $input->getOption('post');
        $force = (bool) $input->getOption('force');

        if ('prod' !== $env) {
            $this->io->warning(\sprintf(
                'No-downtime migrations only make sense in "prod" environment; you are running in "%s".',
                $env
            ));
            if (!$this->io->confirm('Continue?')) {
                return 0;
            }
        }

        if (!$pre && !$post) {
            $this->io->error('You must specify either --pre or --post flags');
        }

        try {
            if ($pre) {
                $this->executePreDeployMigrations($input, $force);
            } elseif ($post) {
                $this->executePostDeployMigrations($input);
            }
        } catch (\Exception $e) {
            $this->io->error($e->getMessage());

            return $e->getCode();
        }

        return 0;
    }

    /**
     * @param InputInterface $input
     * @param bool           $force
     *
     * @throws \Exception
     */
    private function executePreDeployMigrations(InputInterface $input, $force): void
    {
        $currentVersion   = $this->getCurrentVersion();
        $sequenceVersions = \array_reverse($this->sequence['post_deploy_migrations'], true);

        $pendingVersions = [];

        foreach (\array_keys($sequenceVersions) as $preVersion) {
            if ($preVersion <= $currentVersion) {
                //if we executed post deploy migrations already the $preVersion will be less than current.
                break;
            }
            $pendingVersions[] = $preVersion;
        }

        $count = \count($pendingVersions);

        if (0 === $count) {
            $this->io->note('No new pre-deploy migrations found.');

            return;
        }
        if (1 < $count && !$force) {
            throw new Exception(
                'Migrations are too many versions apart. Add --force flag to force execution (should be done in maintenance mode only)',
                $count
            );
        }

        $latestVersion = $pendingVersions[0]; //remove latest version from array.

        // since all migrations are seen by doctrine migrations, as long as we migrate to the latest pre-deploy
        // version, it will ensure execution of all preceeding pre and post deploy migrations leaving only current
        // lastes post-deploy migrations un-executed

        $args = [$latestVersion];
        if ($input->getOption('dry-run')) {
            $args[] = '--dry-run';
        }

        $result = $this->invoke('doctrine:migrations:migrate', $args);
        if ($result) {
            throw new Exception('Failed to execute migrations.', $result);
        }

        $this->io->success('Pre deploy migrations complete');
    }

    /**
     * @param InputInterface $input
     *
     * @throws \Exception
     */
    private function executePostDeployMigrations(InputInterface $input): void
    {
        $currentVersion   = $this->getCurrentVersion();
        $sequenceVersions = \array_reverse($this->sequence['post_deploy_migrations'], true);
        $targetVersion    = null;
        foreach ($sequenceVersions as $preVersion => $postVersions) {
            // find the latest sequence which is of current version or earlier
            if ($preVersion <= $currentVersion) {
                $this->io->success('Chose sequence version: ' . $preVersion);
                foreach ($postVersions as $postVersion) {
                    $targetVersion = \max($targetVersion, $postVersion);
                }
                break;
            }
        }

        if (null === $targetVersion) {
            $this->io->warning('No sequences satisfy current version. Assuming no postDeploy migrations exist for this version');

            return;
        }

        if ($targetVersion === $currentVersion) {
            $this->io->success('All post-deploy migrations are already applied');

            return;
        }

        if ($targetVersion < $currentVersion) {
            $this->io->warning('Database version seems to be ahead of all versions in the sequence file. Make sure your sequence file is up to date.');
            $this->io->note('Nothing new to execute. Quitting.');

            return;
        }

        $this->io->success(\sprintf('Identified "%s" as the target post deploy version', $targetVersion));

        $args = [$targetVersion];
        if ($input->getOption('dry-run')) {
            $args[] = '--dry-run';
        }

        $result = $this->invoke('doctrine:migrations:migrate', $args);
        if ($result) {
            throw new Exception('Failed to execute migrations.', $result);
        }

        $this->io->success('Post deploy migrations complete');
    }

    /**
     * @throws \Exception
     *
     * @return mixed
     *
     * @SuppressWarnings(PHPMD.ElseExpression)
     */
    private function getCurrentVersion()
    {
        $bufferedOutput = new BufferedOutput();
        $result         = $this->invoke('doctrine:migrations:status', ['--no-ansi'], $bufferedOutput);
        if ($result) {
            throw new Exception($bufferedOutput->fetch(), $result);
        }

        $data = $bufferedOutput->fetch();
        if (\preg_match('#\bCurrent Version:.*\((\d+)\)[\r\n]+#i', $data, $matches)) {
        } elseif (\preg_match('#\bCurrent Version:.*\D(0)[\r\n]+#i', $data, $matches)) {
        } else {
            throw new Exception(
                \sprintf(
                    'Could not determine current version from doctrine:migrations:status; the output was:\r\n%s',
                    $data
                ),
                1
            );
        }

        // TODO we must have integration test with doctrine migrations for this, if they change the output this will
        // fail miserably, and probably will need to specify version dependency we are compatible with once we start
        // encountering such failures

        $this->io->success('Determined current version: ' . $matches[1]);

        return $matches[1];
    }
}
