<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\DeploymentBundle\Command;

use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

#[AsCommand('cyber:no-downtime:verify')]
class NoDowntimeVerifyCommand extends AbstractNoDowntimeCommand
{
    protected function configure(): void
    {
        $helpText = <<<EOT
Use this in your CI to avoid deploying bad migrations.

This command makes sure there are no newer pre-deploy migrations than your sequence file specifies.
It also makes sure all version from the sequence file actually exist.
EOT;

        $this->setDescription('Check if there are any issues')
            ->setHelp($helpText);
    }

    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io = new SymfonyStyle($input, $output);

        try {
            $currentState = $this->loadMetadata();
        } catch (\Exception $ex) {
            $io->error($ex->getMessage());

            return self::FAILURE;
        }

        $migrationSequence      = $currentState->getSequence();
        $preDeployVersions      = $currentState->preDeployVersions();
        $postDeployVersions     = $currentState->postDeployVersions();
        $lastSequenceVersion    = $currentState->lastSeqVersion();
        $lastPreDeploy          = $currentState->lastPreDeployVersion();

        if ($lastPreDeploy !== $lastSequenceVersion) {
            $io->error('You a newer pre-deploy version that does not exist in the sequence file: ' . $lastPreDeploy);
            $io->note('You can run "cyber:no-downtime:update" to update the sequence file.');

            return 1;
        }

        $preDeployVersions  = \array_flip($preDeployVersions);
        $postDeployVersions = \array_flip($postDeployVersions);

        $hasErrors = false;
        foreach ($migrationSequence as $preVersion => $postVersions) {
            if (!isset($preDeployVersions[$preVersion])) {
                $io->error(
                    \sprintf('Sequence defines version %s, but corresponding version file was not found.', $preVersion)
                );

                $hasErrors = true;
            }
            foreach ($postVersions as $postVersion) {
                if (!isset($postDeployVersions[$postVersion])) {
                    $io->error(
                        \sprintf(
                            'Sequence defines version %s, but corresponding version file was not found.',
                            $postVersion
                        )
                    );

                    $hasErrors = true;
                }
            }
        }

        if ($hasErrors) {
            return 1;
        }

        $io->success('All Good');

        return 0;
    }
}
