<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\DeploymentBundle\Command;

use Cyber\DeploymentBundle\Configuration;
use Doctrine\ORM\EntityManagerInterface;
use Symfony\Component\Console\Attribute\AsCommand;
use Symfony\Component\Console\Command\Command;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\BufferedOutput;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Console\Style\SymfonyStyle;

/**
 * Class ValidateCommand.
 *
 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
 * @SuppressWarnings(PHPMD.NPathComplexity)
 */
#[AsCommand('cyber:deploy:validate')]
class ValidateCommand extends Command
{
    use AppCommandInvoker;

    public function __construct(private Configuration $config, private EntityManagerInterface $entityManager)
    {
        parent::__construct();
    }

    protected function configure(): void
    {
        $this
            ->setDescription('Validates up and down migrations, with and without fixtures')
            ->addOption(
                'with-prod',
                null,
                InputOption::VALUE_OPTIONAL,
                'Do validation on prod environment as well. If your prod envrionment is not "prod" pass its actual value through this option'
            );
    }

    /**
     * @inheritdoc
     */
    protected function execute(InputInterface $input, OutputInterface $output): int
    {
        $io          = new SymfonyStyle($input, $output);
        $interactive = $input->isInteractive();

        $envs = ['dev'];
        if ($input->hasParameterOption('--with-prod')) {
            $envs[] = (string) ($input->getOption('with-prod') ?? 'prod');
        }

        $connection = $this->entityManager->getConnection();
        $params     = $connection->getParams();
        $host       = $params['host'] ?? 'localhost';
        $dbName     = $connection->getDatabase();

        if ($interactive
            && !$io->confirm(
                \sprintf('This operation will recreated the database "%s@%s" and apply all fixtures.', $dbName, $host),
                false
            )
        ) {
            $io->text('!!!User Aborted!!!');

            return 1;
        }

        $input->setInteractive(false);
        $progress = $io->createProgressBar(100);
        $progress->setRedrawFrequency(1);
        $progress->setMessage('Dropping Schema');
        $progress->start();

        //drop it just in case it exists, usually on dev machines
        $this->invoke('doctrine:schema:drop', ['-q', '--force', '--full-database']);

        $progress->advance(10);
        $step = (int) (16 / \count($envs));

        foreach ($envs as $env) {
            $progress->setMessage('Validating ' . $env);
            //do validation
            if ($this->invoke('doctrine:migrations:migrate', ['-n', '-q', '--env=' . $env])) {
                $io->error('Failed to execute all up migrations');

                return 21;
            }
            $progress->advance($step);
            if ($this->validateSchemaState($io)) {
                $io->error('Failed to validate that schema matches entities after applying all migrations');

                return 22;
            }
            $progress->advance($step);
            if ($this->invoke('doctrine:migrations:migrate', ['-n', '-q', '--env=' . $env, 'first'])) {
                //test the down migrations
                $io->error('Failed to execute all down migrations');

                return 23;
            }
            $progress->advance($step);
            if ($this->validateSchemaState($io, true)) {
                $io->error('Failed to validate that schema is empty after executing all down migrations.');

                return 24;
            }
            $progress->advance($step);
        }

        if ($this->invoke('cyber:deploy:fixtures', ['-n', '-q', '--dev'])) {
            $io->error('Failed to apply dev fixtures');

            return 25;
        }
        $progress->advance(9);
        if ($this->invoke('doctrine:migrations:migrate', ['-n', '-q', 'first'])) {
            $io->error('Failed to apply all down migrations when tables contain data from fixtures.');

            return 26;
        }
        $progress->advance(9);
        if ($this->invoke('doctrine:schema:drop', ['-q', '--force', '--full-database'])) {
            $io->error('Failed to drop schema after completion.');

            return 27;
        }
        $progress->advance(8);

        return 0;
    }

    /**
     * @param SymfonyStyle $io
     * @param bool         $empty
     *
     * @return int
     *
     * @SuppressWarnings(PHPMD.BooleanArgumentFlag)
     */
    private function validateSchemaState(SymfonyStyle $io, $empty = false): int
    {
        if ($empty) {
            //validate empty schema
            $schemaManager = $this->entityManager->getConnection()->createSchemaManager();
            $tables        = $schemaManager->listTableNames();
            foreach ($tables as $table) {
                if (\mb_strtolower($table) !== $this->config->getMigrationTable()) {
                    $io->error('Schema is not empty. It has following tables:');
                    $io->listing($tables);

                    return 1;
                }
            }

            return 0;
        }

        //validate non-empty schema
        $tempOutput = new BufferedOutput();
        $exitCode   = $this->invoke(
            'doctrine:schema:update',
            ['--dump-sql', '--complete'],
            $tempOutput
        );
        $outputText = $tempOutput->fetch();
        if ($exitCode) {
            $io->error(\sprintf(
                'Command %s exited with non-zero code: %d',
                'doctrine:schema:update',
                $exitCode
            ));
            $io->error('Additional error information:');
            $io->error($outputText);

            return $exitCode;
        }

        if (false === \mb_strpos($outputText, 'Nothing to update')) {
            $io->error('It seems like migrations are no in sync with the entities. The following differences were identified:');
            $io->block($outputText, null, 'error', ' !  ');

            return 1;
        }

        $exitCode   = $this->invoke('doctrine:schema:validate', ['--skip-sync'], $tempOutput);
        $outputText = $tempOutput->fetch();
        if ($exitCode) {
            $io->error(\sprintf(
                'Command %s exited with non-zero code: %d',
                'doctrine:schema:validate',
                $exitCode
            ));
            $io->error('Additional error information:');
            $io->error($outputText);

            return $exitCode;
        }

        return 0;
    }
}
