<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\DeploymentBundle\Migrations;

use RuntimeException;
use Symfony\Component\Console\Style\SymfonyStyle;
use UnexpectedValueException;

class MigrationMerger
{
    /** @var SymfonyStyle */
    private $io;

    /** @var string */
    private $sourceDir;

    /** @var string */
    private $destDir;

    /**
     * @var string
     */
    private $migrationNamespace;

    /**
     * @param SymfonyStyle $io
     * @param string       $migrationNamespace
     * @param string       $sourceDir
     * @param string       $destDir
     */
    public function __construct(SymfonyStyle $io, string $migrationNamespace, string $sourceDir, string $destDir)
    {
        $this->io                 = $io;
        $this->migrationNamespace = $migrationNamespace;
        $this->sourceDir          = $sourceDir;
        $this->destDir            = $destDir;
    }

    /**
     * @param string $category
     *
     * @throws MergeMigrationException
     *
     * @return null|string
     */
    public function merge(string $category): ?string
    {
        $sourcePath = $this->sourceDir . DIRECTORY_SEPARATOR . $category;
        $destPath   = $this->destDir . DIRECTORY_SEPARATOR . $category;

        $versions = \scandir($sourcePath, SCANDIR_SORT_ASCENDING);
        if (false === $versions) {
            throw new UnexpectedValueException('Failed to read version from path: ' . $sourcePath);
        }

        $versionData = new VersionData();

        foreach ($versions as $version) {
            if (\preg_match('#^(Version\d*)\.php$#', $version, $matches)) {
                $versionName = $matches[1];
                $versionPath = $sourcePath . DIRECTORY_SEPARATOR . $version;
                $this->processVersionFile($versionName, $versionPath, $versionData);
            }
        }

        if (!$versionData->isValid()) {
            throw new MergeMigrationException($versionData->toErrors());
        }

        $functions = $versionData->listVersionFunctions();

        if (empty($functions)) {
            // no new versions
            return null;
        }

        $fullUp = $this->mergeVersionData($functions, 'up');
        //down migrations need to execute in reverse order.
        $fullDown = $this->mergeVersionData(\array_reverse($functions), 'down');

        $newVersion = \date('YmdHis');

        $versionPath = $this->generateProdVersion($destPath, $newVersion, $fullUp, $fullDown, $versionData);

        $this->archiveDevVersions($category, \array_keys($functions), $newVersion);

        $this->checkBestPractices($versionData->listImports(), $fullUp . PHP_EOL . $fullDown);

        return $versionPath;
    }

    /**
     * @param string      $versionName
     * @param string      $versionPath
     * @param VersionData $versionData
     */
    private function processVersionFile($versionName, $versionPath, VersionData $versionData): void
    {
        $this->io->note('Processing ' . $versionName);
        $data = \file_get_contents($versionPath);
        if (false === $data) {
            throw new RuntimeException('Failed to read version file: ' . $versionPath);
        }

        $this->gatherSharedData($data, $versionData);

        $mergeErrors = $versionData->toErrors();

        //check if we have pre or post up/down function as those are not supported for merged migrations
        $prePostPattern = '#function (pre|post)(Up|Down)\s*\(.+\)#';

        if (\preg_match($prePostPattern, $data, $matches)) {
            $mergeErrors->addError($matches[0] . ' in file ' . $versionPath . ' is not supported by this merge utility.');
        }

        //extract all up() and down()
        $upPattern   = '#function up\(.+\)#';
        $downPattern = '#function down\(.+\)#';

        if (!\preg_match($upPattern, $data, $matches, PREG_OFFSET_CAPTURE)) {
            $mergeErrors->addError('Function up() not found in ' . $versionPath);
        }
        $upOffset = $matches[0][1] ?? -1;

        if (!\preg_match($downPattern, $data, $matches, PREG_OFFSET_CAPTURE)) {
            $mergeErrors->addError('Function down() not found in ' . $versionPath);
        }
        $downOffset = $matches[0][1] ?? -1;

        if (0 === $mergeErrors->count()) {
            //do not bother parsing up/down functions if we have any errors
            $extractor = new FunctionExtractor($data);

            try {
                $upData   = $extractor->getFunctionData($upOffset);
                $downData = $extractor->getFunctionData($downOffset);

                //only set the data if we did not get exception during both of above calls
                $versionData->addVersionFunctions($versionName, $upData, $downData);
            } catch (\UnexpectedValueException $ex) {
                $mergeErrors->addError($ex->getMessage() . ' in ' . $versionPath);
            }
        }
    }

    private function gatherSharedData(string $data, VersionData $versionData): void
    {
        //extract all 'use' statements, they start at beginning of the line
        \preg_match_all('/^use .*;$/m', $data, $usages);

        foreach ($usages[0] as $usage) {
            $versionData->addImport($usage);
        }

        // extract all trait 'use' statements, they have some space before them
        \preg_match_all('/^[ ]+use .*;$/m', $data, $traitUsages);

        foreach ($traitUsages[0] as $trait) {
            $versionData->addTrait($trait);
        }

        //extract all 'const' statements
        \preg_match_all('/^\s*const\s*([_A-z0-9]+)\s*=\s*(.*?);.*?$/sm', $data, $constGroups);

        try {
            $constants = \array_combine($constGroups[1], $constGroups[2]);
        } catch (\Throwable $ex) {
            // as of php 8 this throws exception instead of returning false
            $constants = false;
        }
        if (false === $constants) {
            throw new RuntimeException('Failed to parse constants, got inconsistent arrays: ' . \var_export(
                $constGroups,
                true
            ));
        }

        foreach ($constants as $key => $value) {
            $versionData->addConst($key, $value);
        }
    }

    /**
     * @param array<array<string,string>> $versionData
     * @param string                      $direction
     *
     * @return string
     */
    private function mergeVersionData(array $versionData, string $direction): string
    {
        $spacing  = \str_repeat(' ', 8);
        $fullData = '';

        foreach ($versionData as $name => $data) {
            $fullData .= PHP_EOL . $spacing . '// ' . $direction . ' data from ' . $name . PHP_EOL;
            $fullData .= $data[$direction] . PHP_EOL;
            $fullData .= $spacing . '// ==================================================' . PHP_EOL;
        }

        return $fullData;
    }

    /** @noinspection MoreThanThreeArgumentsInspection
     * @param string      $destPath
     * @param string      $newVersion
     * @param string      $upData
     * @param string      $downData
     * @param VersionData $versionData
     *
     * @return string
     */
    private function generateProdVersion($destPath, $newVersion, $upData, $downData, VersionData $versionData): string
    {
        $constants = \array_reduce($versionData->listConstants(), function ($prev, $next) {
            return $prev . $next . PHP_EOL;
        }, '');

        if (0 < \mb_strlen($constants)) {
            $constants .= PHP_EOL;
        }

        $usages = \array_reduce($versionData->listImports(), function ($prev, $next) {
            return $prev . $next . PHP_EOL;
        }, '');

        $traits = \array_reduce($versionData->listTraits(), function ($prev, $next) {
            return $prev . $next . PHP_EOL;
        }, '');

        if (0 < \mb_strlen($traits)) {
            $traits .= PHP_EOL;
        }

        $className = 'Version' . $newVersion;
        $migPath   = $destPath . DIRECTORY_SEPARATOR . $className . '.php';

        /** @noinspection HtmlUnknownTag */
        $migrationTemplate = <<<EOT
<?php declare(strict_types=1);

namespace <namespace>;

<usages>
/**
 * Auto-generated Migration: Please modify to your needs!
 */
final class <class> extends AbstractMigration
{
<traits><constants>    /**
     * {@inheritdoc}
     */
    public function up(Schema \$schema): void
    {<up>    }

    /**
     * {@inheritdoc}
     */
    public function down(Schema \$schema): void
    {<down>    }
}

EOT;

        /** @noinspection HtmlUnknownTag */
        $migration = \strtr($migrationTemplate, [
            '<namespace>' => $this->migrationNamespace,
            '<usages>'    => $usages,
            '<traits>'    => $traits,
            '<constants>' => $constants,
            '<class>'     => $className,
            '<up>'        => $upData,
            '<down>'      => $downData,
        ]);

        if (false === \file_put_contents($migPath, $migration)) {
            $error = \error_get_last();
            $error = $error ? $error['message'] : '';
            throw new RuntimeException('Failed to write migration file: ' . $migPath . ' ' . $error);
        }

        $realPath = \realpath($migPath);
        if (false === $realPath) {
            throw new RuntimeException('Failed to determine real path for migration: ' . $migPath);
        }

        return $realPath;
    }

    /**
     * @param string   $fullUpDown
     * @param string[] $usages
     *
     * @SuppressWarnings(PHPMD.ElseExpression)
     */
    private function checkBestPractices($usages, $fullUpDown): void
    {
        foreach ($usages as $fqcn) {
            switch (true) {
                case \preg_match('#^use Doctrine\\\\DBAL\\\\(.*);$#', (string) $fqcn, $matches):
                    // DBAL namespace should be acceptable too except for Connection
                    if ('Connection' === $matches[1]) {
                        $this->io->warning('Migrations should not use connection directly, make sure it is really necessary');
                    }
                    break;
                case \preg_match('#^use Doctrine\\\\Migrations\\\\(.*);$#', (string) $fqcn, $matches):
                    // everything from migrations namespace is ok.
                    break;
                case 'use Cyber\DeploymentBundle\MigrationHelpers;' === (string) $fqcn:
                    // our own migration helpers are ok
                    break;
                default:
                    // if there are any other usages create a warning
                    $this->io->warning('Migrations should not depend any project related classes and or constants');
            }
        }

        if (\preg_match('#\\$this->connection(?!->getDatabasePlatform)#', $fullUpDown)) {
            $this->io->warning('Migrations should not use connection directly, make sure it is really necessary');
        }
    }

    /**
     * @param string            $category
     * @param array<int|string> $versions
     * @param string            $newVersion
     */
    private function archiveDevVersions(string $category, array $versions, $newVersion): void
    {
        $dest = $this->sourceDir . DIRECTORY_SEPARATOR . $category . DIRECTORY_SEPARATOR . $newVersion;
        if (!@\mkdir($dest) && !@\is_dir($dest)) {
            throw new RuntimeException(\sprintf('Directory "%s" was not created', $dest));
        }

        foreach ($versions as $ver) {
            $filename = DIRECTORY_SEPARATOR . $ver . '.php';
            $src      = $this->sourceDir . DIRECTORY_SEPARATOR . $category . $filename;
            \rename($src, $dest . $filename);
        }
    }
}
