<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\DeploymentBundle\Migrations;

use UnexpectedValueException;

class FunctionExtractor
{
    /** @var string */
    private $data;

    /**
     * @param string $data
     */
    public function __construct(string $data)
    {
        $this->data = $data;
    }

    public function getFunctionData(int $headingOffset): string
    {
        $funcStar = $this->getStartOfFunction($headingOffset) + 1;
        $funcEnd  = $this->getEndOfFunction($funcStar) - 1;

        return \rtrim(\mb_substr($this->data, $funcStar, $funcEnd - $funcStar));
    }

    /**
     * @param int $headingOffset
     *
     * @return int
     */
    private function getStartOfFunction($headingOffset): int
    {
        $startPos = \mb_strpos($this->data, '{', $headingOffset);

        if (false === $startPos) {
            throw new UnexpectedValueException('Failed to find start of function');
        }

        return $startPos;
    }

    /**
     * @param int $startOffset
     *
     * @return int
     */
    private function getEndOfFunction($startOffset): int
    {
        do {
            if (!\preg_match('#[{}]#', $this->data, $matches, PREG_OFFSET_CAPTURE, $startOffset)) {
                throw new UnexpectedValueException('Failed to find end of function');
            }

            $pos = $matches[0][1];

            $char = $this->data[$pos];

            if ('}' === $char) {
                break;
            }
            //in case if we have internal blocks {} inside the function find end of the internal block and set as new start offset
            $startOffset = $this->getEndOfFunction($pos + 1) + 1;
        } while (true);

        return $pos;
    }
}
