<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Tests\Cyber\DeploymentBundle;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Platforms\AbstractPlatform;
use Doctrine\DBAL\Result;
use PHPUnit\Framework\TestCase;

/**
 * @internal
 *
 * @coversNothing
 */
class MigrationHelpersTest extends TestCase
{
    /** @var MockMigrationHelper */
    private $helper;

    /**
     * @var Connection|\PHPUnit\Framework\MockObject\MockObject
     */
    private $mockConnection;

    public function testGetUuid4Sql(): void
    {
        $actual = $this->helper->getUuid4Sql();
        static::assertEquals(4, \mb_substr_count($actual, '-'));
        static::assertGreaterThan(0, \mb_strpos($actual, 'uuid4'));

        $actual = $this->helper->getUuid4Sql('somename', '_');
        static::assertEquals(4, \mb_substr_count($actual, '_'));
        static::assertGreaterThan(0, \mb_strpos($actual, 'somename'));
    }

    public function testGetUuid4BinSql(): void
    {
        $actual = $this->helper->getUuid4BinSql();
        static::assertStringContainsString('UNHEX', $actual);
        static::assertEquals(0, \mb_substr_count($actual, '-'));
        static::assertGreaterThan(0, \mb_strpos($actual, 'uuid4'));

        $actual = $this->helper->getUuid4BinSql('otherName');
        static::assertStringContainsString('UNHEX', $actual);
        static::assertEquals(0, \mb_substr_count($actual, '-'));
        static::assertGreaterThan(0, \mb_strpos($actual, 'otherName'));
    }

    public function testSomething(): void
    {
        $statements = [
            'SET foreign_key_checks=OFF',
            'ALTER TABLE src ADD CONSTRAINT FK FOREIGN KEY (col1) REFERENCES trg (col2) , ALGORITHM = INPLACE',
            'SET foreign_key_checks=ON',
            'SET foreign_key_checks=OFF',
            'ALTER TABLE src ADD CONSTRAINT FK FOREIGN KEY (col1) REFERENCES trg (col2) ON DELETE CASCADE, ALGORITHM = INPLACE',
            'SET foreign_key_checks=ON',
        ];

        $this->helper->concurrentAddForeignKey('FK', 'src', 'trg', 'col1', 'col2');
        $this->helper->concurrentAddForeignKey('FK', 'src', 'trg', 'col1', 'col2', 'CASCADE');

        $sql = $this->helper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }
    }

    public function testConcurrentAddIndex(): void
    {
        $statements = [
            'CREATE INDEX test_index ON alerts (alt_internal, alt_external) ALGORITHM = INPLACE',
            'CREATE UNIQUE INDEX test_index ON alerts (alt_internal, alt_external) ALGORITHM = INPLACE',
        ];

        $this->helper->concurrentAddIndex('alerts', 'test_index', ['alt_internal', 'alt_external']);
        $this->helper->concurrentAddIndex('alerts', 'test_index', ['alt_internal', 'alt_external'], true);

        $sql = $this->helper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }
    }

    public function testConcurrentRemoveIndex(): void
    {
        $statements = ['DROP INDEX test_index ON alerts ALGORITHM = INPLACE'];

        $this->helper->concurrentRemoveIndex('alerts', 'test_index');

        $sql = $this->helper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }
    }

    public function testUpdatedInBatches(): void
    {
        $resultStm = $this->getMockBuilder(Result::class)->disableOriginalConstructor()->getMock();

        $this->mockConnection->expects(static::once())
            ->method('executeQuery')
            ->willReturn($resultStm);

        $resultStm->expects(static::once())
            ->method('fetchAssociative')
            ->willReturn(['ct' => 5]);

        $statements = [
            'CREATE INDEX tmp_idx_1999dbe27c13 ON alerts (message) ALGORITHM = INPLACE',
            'UPDATE alerts SET message = \'test message\' WHERE message != \'test message\' OR \'test message\' IS NULL AND message IS NOT NULL LIMIT 1000',
            'UPDATE alerts SET message = \'test message\' WHERE message != \'test message\' OR \'test message\' IS NULL AND message IS NOT NULL',
            'DROP INDEX tmp_idx_1999dbe27c13 ON alerts ALGORITHM = INPLACE',
        ];

        $this->helper->updateColumnInBatches('alerts', 'message', '\'test message\'');

        $sql = $this->helper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }
    }

    /**
     * @SuppressWarnings("PMD.UnusedLocalVariable")
     */
    protected function setUp(): void
    {
        $mockPlatform         = $this->getMockBuilder(AbstractPlatform::class)->getMock();
        $this->mockConnection = $this->getMockBuilder(Connection::class)->disableOriginalConstructor()->getMock();

        $mockPlatform->method('quoteIdentifier')
            ->willReturnArgument(0);

        $this->helper = new MockMigrationHelper($mockPlatform, $this->mockConnection);
    }
}
