*[Home](index.md) › Migration Helpers*

# Overview
This bundle provides a `Cyber\DeploymentBundle\MigrationHelpers` trait that you can use in your migration classes. The
helpers are a collection of functions that assist in altering tables in such a way as to reduce the need for downtime.

They also provide a few other functions that might be helpful in some situations.

The sections below will reference these helpers when discussing migrations.

# UUID Helpers
MySql's UUID() function creates UUIDs of v1 type. These helpers provide a way to generate UUID v4 in pure SQL.
This could be used when moving from AUTO-INCREMENT to UUID identities.

## Human UUID
The `MigrationHelpers::getUuid4Sql()` generates a human readable UUID v4. You can customize column alias and separators
via arguments. Setting separator to empty string will result in just list of hex digits.

## Binary UUID
This will probably be more useful in migrations as your UUID column will be binary, and you will need to insert binary
id during migration. For example:

```php
<?php

/** @var Cyber\DeploymentBundle\MigrationHelpers $helper */
$helper = null;

$this->addSql('
INSERT INTO uuid_table (id, col_1, col_2, col_3)  
    SELECT ' . $helper->getUuid4BinSql() . ', col_1, col_2, col_3 FROM source_table
');
```

# What Requires Downtime?

When working with a database certain operations can be performed without taking your application offline, others do
require a downtime period. This guide describes various operations, their impact, and how to perform them **WITHOUT**
requiring downtime.


## Adding Columns
You can safely add a new column to an existing table as long as it does not have a default value. For example, this 
query would not require downtime:
```sql
ALTER TABLE projects ADD COLUMN random_value int;
```

Add a column with a default however does require downtime. For example, consider this query:
```sql
ALTER TABLE projects ADD COLUMN random_value int DEFAULT 42;
```

This requires updating every single row in the projects table so that random_value is set to 42 by default. This 
requires updating all rows and indexes in a table. This in turn acquires enough locks on the table for it to effectively
block any other queries.

> As of MySQL `5.7` adding a column to a table is still quite an expensive operation, even when using 
> `ALGORITHM=INPLACE` and `LOCK=NONE`. This means downtime may be required when modifying large tables as otherwise the 
> operation could potentially take hours to complete.
>
> Still the operation is faster without the default value by about 15% - 20%

Adding a column with a default value can be done without requiring downtime when using the migration helper method 
`MigrationHelpers::concurrentAddColumnWithDefault()`. For example:

```php
<?php
use Cyber\DeploymentBundle\MigrationHelpers;
use Doctrine\DBAL\Schema\Schema;
use Doctrine\Migrations\AbstractMigration;

class VersionXYZ extends AbstractMigration {
    use MigrationHelpers;
    
    public function up(Schema $schema): void
    {
        $this->concurrentAddColumnWithDefault('projects', 'random_value', 'VARCHAR(50)', 'unknown');
    }
    
    public function down(Schema $schema): void
    {
        $this->addSql('ALTER TABLE projects DROP random_value');
    }
}
```

Keep in mind that even though it does not cause downtime, this operation can take long time to complete on large enough 
tables. As a result you should only add default values if absolutely necessary. 

## Dropping Columns
Removing columns is a bit tricky because running application process may still be using the column while new code will 
insert `null` values for all new rows. To work around this the migration needs to be completed in 2 steps: one to set
the column to allow nulls, and one to remove the column. 

### Step 1: Allow Null
The new application code will start inserting rows into the table without providing value for the column to be removed. 
So it needs to accept null values if it did not do so before. 

This step needs to happen in pre-deploy migration (before new application code goes live)

```sql
-- PRE DEPLOY MIGRATION
ALTER TABLE projects CHANGE random_value random_value INT DEFAULT NULL;
```

### Step 2: Drop Column
The column needs to remain in the table as long as the old application code is running, as it may still be selecting
that column through *SELECT ALL FIELDS* queries, even though the field is not used in code.

So we remove the column in the post-deploy migration.

```sql
-- POST DEPLOY MIGRATION
ALTER TABLE projects DROP random_value;
```
 
## Renaming Columns
Renaming columns the normal way requires downtime as an application may continue using the old column name during/after 
a database migration. Similar to **Dropping Columns**,  in order to rename a column without requiring downtime we need
pre- and post-deployment migrations.

### Step 1: Add Column With New Name
Create the pre-deploy migration using the helper `MigrationHelpers::concurrentRenameColumn()`. For example:

```php
<?php
use Cyber\DeploymentBundle\MigrationHelpers;
use Doctrine\DBAL\Schema\Schema;
use Doctrine\Migrations\AbstractMigration;

class VersionXYZ extends AbstractMigration {
    use MigrationHelpers;
    
    public function up(Schema $schema): void
    {
        $this->concurrentRenameColumn($schema, 'projects', 'random_value', 'secret_value');
        // ADD any needed indexes and FKs that need to exist on this column
    }
    
    public function down(Schema $schema): void
    {
        $this->cleanupConcurrentColumnRename('projects', 'secret_value', 'random_value');
    }
}
```

This will take care of renaming the column and ensuring data sync. However, you still need to add SQL for creating indexes
and foreign keys on the new column.

> NOTE: the down migration has column values in reverse. It is because this migration in `down` direction is expected to
> only execute after the post-deploy migration `down` direction has been executed. So you will not be able to `up` and
> `down` this migration alone, without coupling it with the downtime counterpart.

### Step 2: Remove old column
The remaining part requires just some cleaning up in post-deployment migration. Again the helper 
`MigrationHelpers::cleanupConcurrentColumnRename()` comes to the rescue:

```php
<?php
use Cyber\DeploymentBundle\MigrationHelpers;
use Doctrine\DBAL\Schema\Schema;
use Doctrine\Migrations\AbstractMigration;

class VersionXYZ extends AbstractMigration {
    use MigrationHelpers;
    
    public function up(Schema $schema): void
    {
        $this->cleanupConcurrentColumnRename('projects', 'random_value', 'secret_value');
    }
    
    public function down(Schema $schema): void
    {
        $this->concurrentRenameColumn($schema, 'projects', 'secret_value', 'random_value');
        // ADD any needed indexes and FKs that need to exist on this column
    }
}
```

> NOTE: in `up` and `down` part fields are in same order as in the corresponding pre-deployment migration.

## Changing Column Constraints
Adding or removing a *NOT NULL* clause (or another constraint) can typically be done without requiring downtime.
However, this needs to be done in a proper migration (pre or post) depending on how the constraint is changing.
The table below summarizes the change and when it should be done:

| Pre Deploy        | Post Deploy    |
| ----------------- | -------------- |
| REMOVE *NOT NULL* | ADD *NOT NULL* |
| REMOVE *UNIQUE*   | ADD *UNIQUE*   |

In general adding any additional restriction on column should be done in POST-deploy migration after ensuring that
all data meets the stricter requirements. Removing the constraint should be done in PRE-deploy migration, since new
code could start adding data that violates the existing constraint immediately.

## Changing Column Types
Changing column type works similarly to renaming and requires 2 migrations to accomplish it without downtime. For 
example, let's say we want to change type of `user.bio` from `varchar` to `text`

### Step 1: Add Column With New Name
Create the pre-deploy migration using the helper `MigrationHelpers::concurrentChangeColumnType()`. For example:

```php
<?php
use Cyber\DeploymentBundle\MigrationHelpers;
use Doctrine\DBAL\Schema\Schema;
use Doctrine\Migrations\AbstractMigration;

class VersionXYZ extends AbstractMigration {
    use MigrationHelpers;
    
    public function up(Schema $schema): void
    {
        $this->concurrentChangeColumnType($schema, 'user', 'bio', 'TEXT');
    }
    
    public function down(Schema $schema): void
    {
        $this->cleanupConcurrentChangeColumnType($schema, 'user', 'bio');
        // ADD any needed indexes and FKs that need to exist on this column
    }
}
```

### Step 2: Remove old column
The remaining part requires just some cleaning up in post-deployment migration. Again the helper 
`MigrationHelpers::cleanupConcurrentChangeColumnType()` comes to the rescue:

```php
<?php
use Cyber\DeploymentBundle\MigrationHelpers;
use Doctrine\DBAL\Schema\Schema;
use Doctrine\Migrations\AbstractMigration;

class VersionXYZ extends AbstractMigration {
    use MigrationHelpers;
    
    public function up(Schema $schema): void
    {
        $this->cleanupConcurrentChangeColumnType($schema, 'user', 'bio');
        // ADD any needed indexes and FKs that need to exist on this column
    }
    
    public function down(Schema $schema): void
    {
        $this->concurrentChangeColumnType($schema, 'user', 'bio', 'VARCHAR(10)');
    }
}
```

**NOTE**: the indexes and foreign keys will be lost on the column, you need to re-add them here

## Adding Indexes
Adding indexes is an expensive process that blocks INSERT and UPDATE queries for the duration. As a workaround you can
add an `ALGORITHM = INPLACE` flag to your query to make it non-blocking.

```sql
-- up
CREATE INDEX id_index ON `user` (usr_username) ALGORITHM = INPLACE;

-- down
DROP INDEX id_index ON `user` ALGORITHM = INPLACE;
```

> Helper functions: `concurrentAddIndex($table, $name, $columns, $unique)`, `concurrentRemoveIndex($table, $name)`

## Dropping Indexes
Dropping an index does not require downtime.

## Adding Tables
This operation is safe as there’s no code using the table just yet.

## Dropping Tables
Dropping tables can be done safely using a post-deployment migration.

## Adding Foreign Keys
Adding foreign keys usually works in 3 steps:

1. Start a transaction
2. Run ALTER TABLE to add the constraint(s)
3. Check all existing data

Because ALTER TABLE typically acquires an exclusive lock until the end of a transaction this means this approach would 
require downtime.

You can work around this using `MigrationHelpers::concurrentAddForeignKey()`

> **WARNING**: To achieve no-downtime execution of this query, the foreign key checks are disabled. You have to make 
> sure none of the values in table violate the foreign key, otherwise you may see unexpected issues.

## Removing Foreign Keys
This operation does not require downtime.

## Data Migrations
Data migrations can be tricky. The usual approach to migrate data is to take a 3 step approach:

1. Migrate the initial batch of data
2. Deploy the application code
3. Migrate any remaining data

Usually this works, but not always. For example, if a field’s format is to be changed from JSON to something else we 
have a bit of a problem. If we were to change existing data before deploying application code we’ll most likely run into
errors. On the other hand, if we were to migrate after deploying the application code we could run into the same 
problems.

If you merely need to correct some invalid data, then a post-deployment migration is usually enough. If you need to 
change the format of data (e.g. from JSON to something else) it’s typically best to add a new column for the new data 
format, and have the application use that. In such a case the procedure would be:

1. Add a new column in the new format
2. Copy over existing data to this new column
3. Deploy the application code
4. In a post-deployment migration, copy over any new data that may have entered db during deploy

In general there is no one-size-fits-all solution, therefore it’s best to discuss these kind of migrations in a merge 
request to make sure they are implemented in the best way possible

