<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\DeploymentBundle;

use Doctrine\DBAL\Platforms\MySQLPlatform;
use Doctrine\DBAL\Platforms\PostgreSQLPlatform;
use Doctrine\DBAL\Schema\Schema;

/**
 * @property \Doctrine\DBAL\Connection                 $connection
 * @property \Doctrine\DBAL\Platforms\AbstractPlatform $platform
 *
 * @method addSql($sql, array $params = [], array $types = []) from migration Version class
 * @method warnIf($condition, $message = '')                   from migration Version class
 */
trait MigrationHelpers
{
    /**
     * Adds column to a table.
     *
     * This is not safe to use in pre-deploy migrations, you should use concurrentAddColumnWithDefault
     *
     * @param string  $table   table name
     * @param string  $column  column name
     * @param string  $type    column type (ex. VARCHAR(10))
     * @param ?string $default column default value, you must quote string literals
     * @param bool    $notNull
     *
     * @SuppressWarnings("PMD.BooleanArgumentFlag")
     */
    public function addColumn(
        string $table,
        string $column,
        string $type,
        ?string $default,
        bool $notNull = false
    ): void {
        $sql = \sprintf(
            'ALTER TABLE %s ADD %s %s %s%s',
            $this->platform->quoteIdentifier($table),
            $this->platform->quoteIdentifier($column),
            $type,
            $this->parseDefault($default, $notNull),
            $this->platform instanceof MySQLPlatform ? ', ALGORITHM = INPLACE' : '',
        );

        $this->addSql($sql);
    }

    /**
     * Changes column in a table.
     *
     * This is not safe to use in pre-deploy migrations, you should use one of concurrent methods. If none suite
     * your need then probably this one will be ok.
     *
     * @param string $table   table name
     * @param string $column  column name
     * @param string $type    column type (ex. VARCHAR(10))
     * @param mixed  $default column default value, you must quote string literals
     * @param bool   $notNull
     *
     * @SuppressWarnings("PMD.BooleanArgumentFlag")
     */
    public function changeColumn(string $table, string $column, string $type, $default, bool $notNull = false): void
    {
        if ($this->platform instanceof PostgreSQLPlatform) {
            $this->addSql(\sprintf(
                'ALTER TABLE %s ALTER %s DROP DEFAULT',
                $this->platform->quoteIdentifier($table),
                $this->platform->quoteIdentifier($column)
            ));

            $this->addSql(\sprintf(
                'ALTER TABLE %s ALTER %s TYPE %s',
                $this->platform->quoteIdentifier($table),
                $this->platform->quoteIdentifier($column),
                $type,
            ));

            $this->addSql(\sprintf(
                'ALTER TABLE %s ALTER %s %s NOT NULL',
                $this->platform->quoteIdentifier($table),
                $this->platform->quoteIdentifier($column),
                $notNull ? 'SET' : 'DROP',
            ));

            return;
        }
        $this->addSql(\sprintf(
            'ALTER TABLE %s MODIFY %s %s %s %s%s',
            $this->platform->quoteIdentifier($table),
            $this->platform->quoteIdentifier($column),
            $type,
            $this->parseDefault($default, $notNull),
            $notNull ? 'NOT NULL' : '',
            $this->platform instanceof MySQLPlatform ? ', ALGORITHM = INPLACE' : '',
        ));
    }

    /**
     * Renames database column.
     *
     * NOTE SAFE FOR PRE_DEPLOY MIGRATIONS, USE concurrentRenameColumn() INSTEAD
     *
     * @param Schema $schema
     * @param string $table
     * @param string $column
     * @param string $newName
     *
     * @throws \Doctrine\DBAL\Schema\SchemaException
     * @throws \Doctrine\DBAL\Exception
     */
    public function renameColumn(Schema $schema, string $table, string $column, string $newName): void
    {
        $sCol = $schema->getTable($table)->getColumn($column);

        $definition = $this->platform->getColumnDeclarationSQL($newName, $sCol->toArray());

        $table  = $this->platform->quoteIdentifier($table);
        $column = $this->platform->quoteIdentifier($column);

        $this->addSql(\sprintf(
            'ALTER TABLE %s CHANGE %s %s%s',
            $table,
            $column,
            $definition,
            $this->platform instanceof MySQLPlatform ? ', ALGORITHM = INPLACE' : '',
        ));
    }

    public function removeColumn(string $table, string $column): void
    {
        $table  = $this->platform->quoteIdentifier($table);
        $column = $this->platform->quoteIdentifier($column);

        $this->addSql(\sprintf(
            'ALTER TABLE %s DROP %s%s',
            $table,
            $column,
            $this->platform instanceof MySQLPlatform ? ', ALGORITHM = INPLACE' : '',
        ));
    }

    public function updateColumnInBatches(string $table, string $column, string $value): void
    {
        // add temporary index for column update
        $indexName = $this->getTempIndexName($table, $column);
        $this->concurrentAddIndex($table, $indexName, [$column]);

        $tableQuoted = $this->platform->quoteIdentifier($table);
        $column      = $this->platform->quoteIdentifier($column);

        /** @var array{ct: int} $result */
        $result = $this->connection->executeQuery('SELECT count(*) as ct FROM ' . $tableQuoted)->fetchAssociative();

        $max = \ceil($result['ct'] / 1000.0);

        for ($i = 0; $i < $max; ++$i) {
            // if value is not same. If column is null it will not be checked using '!=' so need to do it explicitly
            $sql = \sprintf(
                'UPDATE %s SET %s = %s WHERE %s != %s OR %s IS NULL AND %s IS NOT NULL LIMIT 1000',
                $tableQuoted,
                $column,
                $value,
                $column,
                $value,
                $value,
                $column
            );

            $this->addSql($sql);
        }

        // capture any remaining ones
        $sql = \sprintf(
            'UPDATE %s SET %s = %s WHERE %s != %s OR %s IS NULL AND %s IS NOT NULL',
            $tableQuoted,
            $column,
            $value,
            $column,
            $value,
            $value,
            $column
        );

        $this->addSql($sql);
        $this->concurrentRemoveIndex($table, $indexName);
    }

    /**
     * @param string $table
     * @param string $column
     * @param string $type
     * @param string $default you must quote string literal
     * @param bool   $notNull
     *
     * @SuppressWarnings("PMD.BooleanArgumentFlag")
     */
    public function concurrentAddColumnWithDefault(
        string $table,
        string $column,
        string $type,
        string $default,
        bool $notNull = true
    ): void {
        // create the column with default value as NULL, this operation does not require full table update so it
        // happens quickly
        $this->addColumn($table, $column, $type, null);

        // update the default value, so all newly inserted rows are inserted with this new value.
        $this->changeColumn($table, $column, $type, $default);

        // now update all existing records with the default value
        $this->updateColumnInBatches($table, $column, $default);

        if ($notNull) {
            // change column to not be nullable
            $this->changeColumn($table, $column, $type, $default, $notNull);
        }
    }

    public function concurrentChangeColumnType(Schema $schema, string $table, string $column, string $newType): void
    {
        $tempColumn = $this->getRenameColumnName($column);

        $this->concurrentRenameColumn($schema, $table, $column, $tempColumn, $newType);
    }

    /**
     * @throws \Doctrine\DBAL\Exception
     * @throws \Doctrine\DBAL\Schema\SchemaException
     */
    public function cleanupConcurrentChangeColumnType(Schema $schema, string $table, string $column): void
    {
        $tempColumn = $this->getRenameColumnName($column);

        $this->addSql(\sprintf('LOCK TABLES %s WRITE', $this->platform->quoteIdentifier($table)));
        $this->cleanupConcurrentColumnRename($table, $column, $tempColumn);
        $this->renameColumn($schema, $table, $tempColumn, $column);
        $this->addSql('UNLOCK TABLES');
    }

    public function concurrentRenameColumn(
        Schema $schema,
        string $table,
        string $old,
        string $new,
        ?string $newType = null
    ): void {
        $sTable  = $schema->getTable($table);
        $sColumn = $sTable->getColumn($old);
        $options = $sColumn->toArray();

        $autoIncrement            = $options['autoincrement'];
        $options['autoincrement'] = false; // usually only single auto increment is allowed, so make column without it

        $type = $newType ?? $sColumn->getType()->getSQLDeclaration($options, $this->platform);

        $this->warnIf($autoIncrement, 'You should try to avoid AUTO_INCREMENT column changes without downtime');

        // create the column with default value as NULL, this operation does not require full table update so it
        // happens quickly
        $this->addColumn($table, $new, $type, null);

        $this->installRenameTriggers($table, $old, $new);

        // update the default value, so all newly inserted rows are inserted with this new value.
        if ($sColumn->getDefault()) {
            $this->changeColumn($table, $new, $type, $sColumn->getDefault());
        }

        $this->updateColumnInBatches($table, $new, $old);

        if ($sColumn->getNotnull()) {
            $this->changeColumn($table, $new, $type, $sColumn->getDefault(), true);
        }
        // TODO do we copy indexes here?
    }

    /**
     * @param string   $table
     * @param string[] $columns
     *
     * @return string
     */
    public function addUniqueIndex(string $table, array $columns): string
    {
        $name = 'TMP_UNQ_' . \mb_substr(\md5($table . '_' . \implode('_', $columns)), 0, 12);

        $columns = \array_map([$this->platform, 'quoteIdentifier'], $columns);

        $sql = \sprintf(
            'ALTER TABLE %s ADD UNIQUE INDEX %s (%s)',
            $this->platform->quoteIdentifier($table),
            $this->platform->quoteIdentifier($name),
            \implode(', ', $columns)
        );

        $this->addSql($sql);

        return $name;
    }

    public function cleanupConcurrentColumnRename(string $table, string $old, string $new): void
    {
        $trigger = $this->getRenameTriggerName($table, $old, $new);

        $this->addSql(\sprintf('DROP TRIGGER IF EXISTS %s_insert', $trigger));
        $this->addSql(\sprintf('DROP TRIGGER IF EXISTS %s_update', $trigger));

        $this->removeColumn($table, $old);
    }

    public function concurrentAddForeignKey(
        string $name,
        string $source,
        string $target,
        string $sourceColumn,
        string $targetColumn,
        ?string $onDelete = null
    ): void {
        $name         = $this->platform->quoteIdentifier($name);
        $source       = $this->platform->quoteIdentifier($source);
        $target       = $this->platform->quoteIdentifier($target);
        $sourceColumn = $this->platform->quoteIdentifier($sourceColumn);
        $targetColumn = $this->platform->quoteIdentifier($targetColumn);

        $onDelete = $onDelete ? ' ON DELETE ' . $onDelete : '';

        if ($this->platform instanceof MySQLPlatform) {
            $this->addSql('SET foreign_key_checks=OFF');
        }
        $this->addSql(
            \sprintf(
                'ALTER TABLE %s ADD CONSTRAINT %s FOREIGN KEY (%s) REFERENCES %s (%s)%s%s',
                $source,
                $name,
                $sourceColumn,
                $target,
                $targetColumn,
                $onDelete,
                $this->platform instanceof MySQLPlatform ? ', ALGORITHM = INPLACE' : ' NOT VALID',
            )
        );
        if ($this->platform instanceof MySQLPlatform) {
            $this->addSql('SET foreign_key_checks=ON');
        }
    }

    /**
     * @param string   $table
     * @param string   $name
     * @param string[] $columns
     * @param bool     $unique
     *
     * @SuppressWarnings(PHPMD.BooleanArgumentFlag)
     */
    public function concurrentAddIndex($table, $name, array $columns, $unique = false): void
    {
        $columns = \array_map([$this->platform, 'quoteIdentifier'], $columns);

        $sql = \sprintf(
            'CREATE %s%s %s ON %s (%s)%s',
            ($unique ? 'UNIQUE ' : '') . 'INDEX',
            $this->platform instanceof PostgreSQLPlatform ? ' CONCURRENTLY' : '',
            $this->platform->quoteIdentifier($name),
            $this->platform->quoteIdentifier($table),
            \implode(', ', $columns),
            $this->platform instanceof MySQLPlatform ? ' ALGORITHM = INPLACE' : '',
        );

        $this->addSql($sql);
    }

    public function concurrentRemoveIndex(string $table, string $name): void
    {
        $sql = match(true) {
            $this->platform instanceof PostgreSQLPlatform => \sprintf(
                'DROP INDEX CONCURRENTLY %s',
                $this->platform->quoteIdentifier($name),
            ),
            default => \sprintf(
                'DROP INDEX %s ON %s%s',
                $this->platform->quoteIdentifier($name),
                $this->platform->quoteIdentifier($table),
                $this->platform instanceof MySQLPlatform ? ' ALGORITHM = INPLACE' : '',
            ),
        };

        $this->addSql($sql);
    }

    /**
     * @param string $table
     * @param string $old
     * @param string $new
     *
     * @SuppressWarnings("PMD.UnusedLocalVariable") not detecting using in heredoc
     */
    public function installRenameTriggers(string $table, string $old, string $new): void
    {
        $triggerName = $this->getRenameTriggerName($table, $old, $new);

        $sql = <<<EOT
CREATE TRIGGER {$triggerName}_insert
BEFORE INSERT
ON {$table}
FOR EACH ROW
IF NEW.{$new} IS NULL THEN SET NEW.{$new} = NEW.{$old};
ELSEIF NEW.{$old} IS NULL THEN SET NEW.{$old} = NEW.{$new}; END IF
EOT;

        $this->addSql($sql);

        $sql = <<<EOT
CREATE TRIGGER {$triggerName}_update
BEFORE UPDATE
ON {$table}
FOR EACH ROW
IF NEW.{$old} != OLD.{$old} THEN SET NEW.{$new} = NEW.{$old};
ELSEIF NEW.{$new} != OLD.{$new} THEN SET NEW.{$old} = NEW.{$new};
END IF
EOT;

        $this->addSql($sql);
    }

    public function getRenameColumnName(string $column): string
    {
        return $column . '_for_type_change';
    }

    /**
     * Generates a rename trigger name based on input values.
     *
     * @param string $table table name
     * @param string $old   old column name
     * @param string $new   new column name
     *
     * @return string
     */
    public function getRenameTriggerName(string $table, string $old, string $new): string
    {
        return 'trigger_' . \mb_substr(\md5($table . '_' . $old . '_' . $new), 0, 12);
    }

    public function getTempIndexName(string $table, string $column): string
    {
        return 'tmp_idx_' . \mb_substr(\md5($table . '_' . $column), 0, 12);
    }

    public function getUuid4BinSql(string $columnName = 'uuid4'): string
    {
        return \sprintf('UNHEX(%s) AS %s', $this->getUuid4Sql('', ''), $columnName);
    }

    /**
     * @param string $columnName uuid column alias
     * @param string $separator  set to empty string to get hex values only
     *
     * @return string
     */
    public function getUuid4Sql(string $columnName = 'uuid4', string $separator = '-'): string
    {
        $sql = <<<'EOT'
CONCAT(
    LPAD(HEX(FLOOR(RAND() * 0xffff)), 4, '0'), LPAD(HEX(FLOOR(RAND() * 0xffff)), 4, '0'), '%s',
    LPAD(HEX(FLOOR(RAND() * 0xffff)), 4, '0'), '%s', '4', LPAD(HEX(FLOOR(RAND() * 0x0fff)), 3, '0'), '%s',
    HEX(FLOOR(RAND() * 4 + 8)), LPAD(HEX(FLOOR(RAND() * 0x0fff)), 3, '0'), '%s',
    LPAD(HEX(FLOOR(RAND() * 0xffff)), 4, '0'), LPAD(HEX(FLOOR(RAND() * 0xffff)), 4, '0'),
    LPAD(HEX(FLOOR(RAND() * 0xffff)), 4, '0')
    )
EOT;

        return \sprintf($sql, $separator, $separator, $separator, $separator)
            . ($columnName ? ' AS ' . $columnName : '');
    }

    private function parseDefault(mixed $default, bool $notNull): string
    {
        $default = $default ?? ($notNull ? '' : 'NULL');

        if ('' === $default) {
            return '';
        }

        \assert(\is_scalar($default));

        return 'DEFAULT ' . $default;
    }
}
