<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\DeploymentBundle;

use Symfony\Component\Config\Definition\Configurator\DefinitionConfigurator;
use Symfony\Component\Console\Application;
use Symfony\Component\DependencyInjection\ContainerBuilder;
use Symfony\Component\DependencyInjection\Loader\Configurator\ContainerConfigurator;
use Symfony\Component\HttpKernel\Bundle\AbstractBundle;

class CyberDeploymentBundle extends AbstractBundle
{
    public function configure(DefinitionConfigurator $definition): void
    {
        // @formatter:off
        $definition->rootNode()
            ->children()
                ->arrayNode('fixtures')->info('Affects how Fixture command works. All paths should be relative to %kernel.project_dir%')
                    ->addDefaultsIfNotSet()
                    ->children()
                        ->arrayNode('connections')->prototype('scalar')->info('List of connection names to manage')->end()->end()
                        ->arrayNode('managers')->prototype('scalar')->info('list of entity manager names to manage')->end()->end()
                        ->scalarNode('migration_table')->info('name of migration table, required for validation command')->end()
                        ->booleanNode('isolate_managers')->defaultFalse()
                            ->info('When true executes load command once for each defined manager, else only once with default manager.')
                        ->end()
                    ->end()
                ->end()
                ->arrayNode('migrations')->info('Configuration for merging migrations')
                    ->children()
                        ->scalarNode('dev')->isRequired()->cannotBeEmpty()->info('Path dev migrations to be merged')->end()
                        ->scalarNode('prod')->isRequired()->cannotBeEmpty()->info('Path where to put the merged migration')->end()
                        ->arrayNode('categories')->prototype('scalar')->info('If configured, the commands will look for migrations inside sub-directories with these names instead of "dev" and "prod" directories directly.')->end()->end()
                        ->arrayNode('no_downtime')->info('Configurations for no-downtime migrations')->canBeEnabled()
                            ->children()
                                ->scalarNode('pre_deploy')->isRequired()->info('Path where pre-deployment migrations are located for prod')->end()
                                ->scalarNode('post_deploy')->isRequired()->info('Path where post-deployment migrations are located for prod')->end()
                            ->end()
                        ->end()

                        ->scalarNode('post_deploy_config')->defaultNull()->info('Path to yml file containing doctrine_migrations conf for postDeploy migration storage.')->end()
                    ->end()
                ->end()
                ->arrayNode('log_tagging')->info('Enables adding of tags to all log messages')
                    ->children()
                        ->arrayNode('tags')->info('Key value pairs to be added to logger message "extra" attribute')
                            ->prototype('scalar')
                        ->end()
                    ->end()
                ->end()
            ->end();
        // @formatter:on
    }

    /**
     * @param array{fixtures: mixed[], migrations?: mixed[], log_tagging?: array{tags?: string[]}} $config
     */
    public function loadExtension(array $config, ContainerConfigurator $container, ContainerBuilder $builder): void
    {
        $container->import('../config/services.php');
        if (\class_exists(Application::class)) {
            $container->import('../config/console.php');
        }
        $this->processFixtures($config['fixtures'], $builder);
        if (isset($config['migrations'])) {
            $this->processMigrations($config['migrations'], $builder);
        }

        if (isset($config['log_tagging'], $config['log_tagging']['tags'])) {
            $container->import('../config/log_tagging.php');
            $container->parameters()->set('cyber.deployment.log.tags', $config['log_tagging']['tags']);
        }
    }

    /**
     * @param array<mixed>     $config
     * @param ContainerBuilder $container
     */
    private function processMigrations(array $config, ContainerBuilder $container): void
    {
        $confDef = $container->getDefinition('cyber_deployment.config');

        $confDef->addMethodCall('setMigrationDevPath', [$config['dev']]);
        $confDef->addMethodCall('setMigrationProdPath', [$config['prod']]);
        $confDef->addMethodCall('setMigrationCategories', [$config['categories'] ?? []]);

        if ($config['no_downtime']['enabled']) {
            $this->processNoDowntimeConfig($config['no_downtime'], $container);
        }
    }

    /**
     * @param array<mixed>     $config
     * @param ContainerBuilder $container
     */
    private function processNoDowntimeConfig($config, ContainerBuilder $container): void
    {
        $confDef = $container->getDefinition('cyber_deployment.config');

        $confDef->addMethodCall('setPreDeployPath', [$config['pre_deploy']]);
        $confDef->addMethodCall('setPostDeployPath', [$config['post_deploy']]);
    }

    /**
     * @param array<mixed>     $config
     * @param ContainerBuilder $container
     */
    private function processFixtures($config, ContainerBuilder $container): void
    {
        $confDef = $container->getDefinition('cyber_deployment.config');

        if (isset($config['connections'])) {
            $confDef->addMethodCall('setFixtureConnections', [$config['connections']]);
        }

        if (isset($config['managers'])) {
            $confDef->addMethodCall('setFixtureManagers', [$config['managers']]);
        }

        $confDef->addMethodCall('setMigrationTable', [$config['migration_table'] ?? null]);
        $confDef->addMethodCall('setIsolateManagers', [(bool) $config['isolate_managers']]);
    }
}
