<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Tests\Cyber\DeploymentBundle;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\Platforms\MySQLPlatform;
use Doctrine\DBAL\Platforms\PostgreSQLPlatform;
use Doctrine\DBAL\Result;
use PHPUnit\Framework\TestCase;

/**
 * @internal
 *
 * @covers \Cyber\DeploymentBundle\MigrationHelpers
 */
class MigrationHelpersTest extends TestCase
{
    /** @var MockMigrationHelper */
    private $helper;

    /** @var MockMigrationHelper */
    private $psqlHelper;

    /**
     * @var Connection|\PHPUnit\Framework\MockObject\MockObject
     */
    private $mockConnection;

    public function testGetUuid4Sql(): void
    {
        $actual = $this->helper->getUuid4Sql();
        static::assertEquals(4, \mb_substr_count($actual, '-'));
        static::assertGreaterThan(0, \mb_strpos($actual, 'uuid4'));

        $actual = $this->helper->getUuid4Sql('somename', '_');
        static::assertEquals(4, \mb_substr_count($actual, '_'));
        static::assertGreaterThan(0, \mb_strpos($actual, 'somename'));

        // Test PostgreSQL implementation
        $actual = $this->psqlHelper->getUuid4Sql();
        static::assertEquals('gen_random_uuid() AS uuid4', $actual);

        $actual = $this->psqlHelper->getUuid4Sql('custom_name');
        static::assertEquals('gen_random_uuid() AS custom_name', $actual);

        $actual = $this->psqlHelper->getUuid4Sql('', '_');
        static::assertEquals('gen_random_uuid()', $actual);
    }

    public function testGetUuid4BinSql(): void
    {
        $actual = $this->helper->getUuid4BinSql();
        static::assertStringContainsString('UNHEX', $actual);
        static::assertEquals(0, \mb_substr_count($actual, '-'));
        static::assertGreaterThan(0, \mb_strpos($actual, 'uuid4'));

        $actual = $this->helper->getUuid4BinSql('otherName');
        static::assertStringContainsString('UNHEX', $actual);
        static::assertEquals(0, \mb_substr_count($actual, '-'));
        static::assertGreaterThan(0, \mb_strpos($actual, 'otherName'));

        // Test PostgreSQL implementation - PostgreSQL doesn't have UNHEX function,
        // but it uses the same column name approach
        $actual = $this->psqlHelper->getUuid4BinSql();
        static::assertStringContainsString('UNHEX', $actual);
        static::assertStringContainsString('uuid4', $actual);

        $actual = $this->psqlHelper->getUuid4BinSql('pg_binary_uuid');
        static::assertStringContainsString('UNHEX', $actual);
        static::assertStringContainsString('pg_binary_uuid', $actual);
    }

    public function testConcurrentFQ(): void
    {
        $statements = [
            'SET foreign_key_checks=OFF',
            'ALTER TABLE src ADD CONSTRAINT FK FOREIGN KEY (col1) REFERENCES trg (col2), ALGORITHM = INPLACE',
            'SET foreign_key_checks=ON',
            'SET foreign_key_checks=OFF',
            'ALTER TABLE src ADD CONSTRAINT FK FOREIGN KEY (col1) REFERENCES trg (col2) ON DELETE CASCADE, ALGORITHM = INPLACE',
            'SET foreign_key_checks=ON',
        ];

        $this->helper->concurrentAddForeignKey('FK', 'src', 'trg', 'col1', 'col2');
        $this->helper->concurrentAddForeignKey('FK', 'src', 'trg', 'col1', 'col2', 'CASCADE');

        $sql = $this->helper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }

        $statements = [
            'ALTER TABLE src ADD CONSTRAINT FK FOREIGN KEY (col1) REFERENCES trg (col2) NOT VALID',
            'ALTER TABLE src ADD CONSTRAINT FK FOREIGN KEY (col1) REFERENCES trg (col2) ON DELETE CASCADE NOT VALID',
        ];

        $this->psqlHelper->concurrentAddForeignKey('FK', 'src', 'trg', 'col1', 'col2');
        $this->psqlHelper->concurrentAddForeignKey('FK', 'src', 'trg', 'col1', 'col2', 'CASCADE');

        $sql = $this->psqlHelper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }
    }

    public function testConcurrentAddIndex(): void
    {
        $statements = [
            'CREATE INDEX test_index ON alerts (alt_internal, alt_external) ALGORITHM = INPLACE',
            'CREATE UNIQUE INDEX test_index ON alerts (alt_internal, alt_external) ALGORITHM = INPLACE',
        ];

        $this->helper->concurrentAddIndex('alerts', 'test_index', ['alt_internal', 'alt_external']);
        $this->helper->concurrentAddIndex('alerts', 'test_index', ['alt_internal', 'alt_external'], true);

        $sql = $this->helper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }

        $statements = [
            'CREATE INDEX CONCURRENTLY test_index ON alerts (alt_internal, alt_external)',
            'CREATE UNIQUE INDEX CONCURRENTLY test_index ON alerts (alt_internal, alt_external)',
        ];

        $this->psqlHelper->concurrentAddIndex('alerts', 'test_index', ['alt_internal', 'alt_external']);
        $this->psqlHelper->concurrentAddIndex('alerts', 'test_index', ['alt_internal', 'alt_external'], true);

        $sql = $this->psqlHelper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }
    }

    public function testConcurrentRemoveIndex(): void
    {
        $statements = ['DROP INDEX test_index ON alerts ALGORITHM = INPLACE'];

        $this->helper->concurrentRemoveIndex('alerts', 'test_index');

        $sql = $this->helper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }

        $statements = ['DROP INDEX CONCURRENTLY test_index'];

        $this->psqlHelper->concurrentRemoveIndex('alerts', 'test_index');

        $sql = $this->psqlHelper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }
    }

    public function testAddColumn(): void
    {
        // Test MySQL platform
        $this->helper->addColumn('users', 'email', 'VARCHAR(255)', '\'default@example.com\'', true);
        $sql = $this->helper->getSql();
        static::assertEquals(
            'ALTER TABLE users ADD email VARCHAR(255) DEFAULT \'default@example.com\', ALGORITHM = INPLACE',
            $sql[0]->getStatement()
        );

        // Test PostgreSQL platform
        $this->psqlHelper->addColumn('users', 'email', 'VARCHAR(255)', '\'default@example.com\'', true);
        $sql = $this->psqlHelper->getSql();
        static::assertEquals(
            'ALTER TABLE users ADD email VARCHAR(255) DEFAULT \'default@example.com\'',
            $sql[0]->getStatement()
        );
    }

    public function testChangeColumn(): void
    {
        // Test MySQL platform
        $this->helper->changeColumn('users', 'email', 'VARCHAR(100)', '\'new-default@example.com\'', true);
        $sql = $this->helper->getSql();
        static::assertEquals(
            'ALTER TABLE users MODIFY email VARCHAR(100) DEFAULT \'new-default@example.com\' NOT NULL, ALGORITHM = INPLACE',
            $sql[0]->getStatement()
        );

        // Test PostgreSQL platform
        $this->psqlHelper->changeColumn('users', 'email', 'VARCHAR(100)', '\'new-default@example.com\'', true);
        $sql = $this->psqlHelper->getSql();

        // PostgreSQL uses separate statements
        static::assertEquals('ALTER TABLE users ALTER email DROP DEFAULT', $sql[0]->getStatement());
        static::assertEquals('ALTER TABLE users ALTER email TYPE VARCHAR(100)', $sql[1]->getStatement());
        static::assertEquals('ALTER TABLE users ALTER email SET NOT NULL', $sql[2]->getStatement());

        // Test with NOT NULL set to false
        $this->psqlHelper->changeColumn('users', 'optional_field', 'TEXT', 'NULL', false);
        $sql = $this->psqlHelper->getSql();
        static::assertEquals('ALTER TABLE users ALTER optional_field DROP NOT NULL', $sql[5]->getStatement());
    }

    public function testRemoveColumn(): void
    {
        // Test MySQL platform
        $this->helper->removeColumn('users', 'obsolete_field');
        $sql = $this->helper->getSql();
        static::assertEquals(
            'ALTER TABLE users DROP obsolete_field, ALGORITHM = INPLACE',
            $sql[0]->getStatement()
        );

        // Test PostgreSQL platform
        $this->psqlHelper->removeColumn('users', 'obsolete_field');
        $sql = $this->psqlHelper->getSql();
        static::assertEquals(
            'ALTER TABLE users DROP obsolete_field',
            $sql[0]->getStatement()
        );
    }

    public function testAddUniqueIndex(): void
    {
        // Test MySQL platform
        $indexName = $this->helper->addUniqueIndex('users', ['email', 'username']);

        // Verify the generated index name format
        static::assertStringStartsWith('TMP_UNQ_', $indexName);
        static::assertEquals(20, \mb_strlen($indexName)); // 'TMP_UNQ_' + 12 chars from md5

        $sql = $this->helper->getSql();
        static::assertStringContainsString('ADD UNIQUE INDEX ' . $indexName, $sql[0]->getStatement());
        static::assertStringContainsString('(email, username)', $sql[0]->getStatement());

        // Test PostgreSQL platform
        $pgIndexName = $this->psqlHelper->addUniqueIndex('users', ['email', 'username']);
        $sql         = $this->psqlHelper->getSql();

        static::assertStringStartsWith('TMP_UNQ_', $pgIndexName);
        static::assertStringContainsString('ADD UNIQUE INDEX ' . $pgIndexName, $sql[0]->getStatement());
        static::assertStringContainsString('(email, username)', $sql[0]->getStatement());
    }

    public function testCleanupConcurrentColumnRename(): void
    {
        // Test MySQL platform
        $this->helper->cleanupConcurrentColumnRename('users', 'old_column', 'new_column');
        $sql = $this->helper->getSql();

        $triggerName = $this->helper->getRenameTriggerName('users', 'old_column', 'new_column');
        static::assertEquals("DROP TRIGGER IF EXISTS {$triggerName}_insert", $sql[0]->getStatement());
        static::assertEquals("DROP TRIGGER IF EXISTS {$triggerName}_update", $sql[1]->getStatement());
        static::assertEquals('ALTER TABLE users DROP old_column, ALGORITHM = INPLACE', $sql[2]->getStatement());

        // Test PostgreSQL platform
        $this->psqlHelper->cleanupConcurrentColumnRename('users', 'old_column', 'new_column');
        $sql = $this->psqlHelper->getSql();

        $triggerName  = $this->psqlHelper->getRenameTriggerName('users', 'old_column', 'new_column');
        $functionName = $triggerName . '_func';

        static::assertEquals("DROP TRIGGER IF EXISTS {$triggerName}_insert ON users", $sql[0]->getStatement());
        static::assertEquals("DROP TRIGGER IF EXISTS {$triggerName}_update ON users", $sql[1]->getStatement());
        static::assertEquals("DROP FUNCTION IF EXISTS {$functionName}_insert()", $sql[2]->getStatement());
        static::assertEquals("DROP FUNCTION IF EXISTS {$functionName}_update()", $sql[3]->getStatement());
        static::assertEquals('ALTER TABLE users DROP old_column', $sql[4]->getStatement());
    }

    public function testInstallRenameTriggers(): void
    {
        // Test MySQL platform
        $this->helper->installRenameTriggers('users', 'old_name', 'new_name');
        $sql = $this->helper->getSql();

        $triggerName = $this->helper->getRenameTriggerName('users', 'old_name', 'new_name');

        // Check MySQL format with IF/ELSEIF structure
        static::assertStringContainsString("CREATE TRIGGER {$triggerName}_insert", $sql[0]->getStatement());
        static::assertStringContainsString('BEFORE INSERT', $sql[0]->getStatement());
        static::assertStringContainsString('IF NEW.new_name IS NULL THEN SET NEW.new_name = NEW.old_name;', $sql[0]->getStatement());

        static::assertStringContainsString("CREATE TRIGGER {$triggerName}_update", $sql[1]->getStatement());
        static::assertStringContainsString('BEFORE UPDATE', $sql[1]->getStatement());
        static::assertStringContainsString('IF NEW.old_name != OLD.old_name THEN SET NEW.new_name = NEW.old_name;', $sql[1]->getStatement());

        // Test PostgreSQL platform
        $this->psqlHelper->installRenameTriggers('users', 'old_name', 'new_name');
        $sql = $this->psqlHelper->getSql();

        $triggerName  = $this->psqlHelper->getRenameTriggerName('users', 'old_name', 'new_name');
        $functionName = $triggerName . '_func';

        // Check for PostgreSQL function creation
        static::assertStringContainsString("CREATE OR REPLACE FUNCTION {$functionName}_insert()", $sql[0]->getStatement());
        static::assertStringContainsString('RETURNS TRIGGER', $sql[0]->getStatement());
        static::assertStringContainsString('LANGUAGE plpgsql', $sql[0]->getStatement());

        // Check for PostgreSQL insert trigger
        static::assertStringContainsString("CREATE TRIGGER {$triggerName}_insert", $sql[1]->getStatement());
        static::assertStringContainsString('BEFORE INSERT ON users', $sql[1]->getStatement());
        static::assertStringContainsString("EXECUTE FUNCTION {$functionName}_insert()", $sql[1]->getStatement());

        // Check for PostgreSQL update function
        static::assertStringContainsString("CREATE OR REPLACE FUNCTION {$functionName}_update()", $sql[2]->getStatement());

        // Check for PostgreSQL update trigger
        static::assertStringContainsString("CREATE TRIGGER {$triggerName}_update", $sql[3]->getStatement());
        static::assertStringContainsString('BEFORE UPDATE ON users', $sql[3]->getStatement());
        static::assertStringContainsString("EXECUTE FUNCTION {$functionName}_update()", $sql[3]->getStatement());
    }

    public function testCleanupConcurrentChangeColumnType(): void
    {
        $schema = $this->createMock(\Doctrine\DBAL\Schema\Schema::class);
        $table  = $this->createMock(\Doctrine\DBAL\Schema\Table::class);
        $column = $this->createMock(\Doctrine\DBAL\Schema\Column::class);

        $schema->method('getTable')->willReturn($table);
        $table->method('getColumn')->willReturn($column);
        $column->method('toArray')->willReturn([]);

        // Test MySQL platform
        $this->helper->cleanupConcurrentChangeColumnType($schema, 'users', 'status');
        $sql = $this->helper->getSql();

        $tempColumn = $this->helper->getRenameColumnName('status'); // status_for_type_change

        static::assertEquals('LOCK TABLES users WRITE', $sql[0]->getStatement());
        static::assertStringContainsString('DROP TRIGGER IF EXISTS', $sql[1]->getStatement()); // cleanup triggers
        static::assertStringContainsString("ALTER TABLE users CHANGE {$tempColumn} status VARCHAR(255)", $sql[4]->getStatement()); // rename column
        static::assertEquals('UNLOCK TABLES', $sql[5]->getStatement());

        // Test PostgreSQL platform
        $this->psqlHelper->cleanupConcurrentChangeColumnType($schema, 'users', 'status');
        $sql = $this->psqlHelper->getSql();

        static::assertEquals('LOCK TABLE users IN EXCLUSIVE MODE', $sql[0]->getStatement());
        static::assertStringContainsString('DROP TRIGGER IF EXISTS', $sql[1]->getStatement()); // cleanup triggers
        static::assertStringContainsString('DROP FUNCTION IF EXISTS', $sql[3]->getStatement()); // cleanup functions
        static::assertStringContainsString('ALTER TABLE users DROP status', $sql[5]->getStatement()); // rename column
        static::assertStringContainsString('ALTER TABLE users RENAME COLUMN', $sql[6]->getStatement()); // rename column
    }

    public function testUpdatedInBatches(): void
    {
        $resultStm = $this->getMockBuilder(Result::class)->disableOriginalConstructor()->getMock();

        $this->mockConnection->expects(static::exactly(2))
            ->method('executeQuery')
            ->willReturn($resultStm);

        $resultStm->expects(static::exactly(2))
            ->method('fetchAssociative')
            ->willReturn(['ct' => 5]);

        $statements = [
            'CREATE INDEX tmp_idx_1999dbe27c13 ON alerts (message) ALGORITHM = INPLACE',
            'UPDATE alerts SET message = \'test message\' WHERE message != \'test message\' OR \'test message\' IS NULL AND message IS NOT NULL LIMIT 1000',
            'UPDATE alerts SET message = \'test message\' WHERE message != \'test message\' OR \'test message\' IS NULL AND message IS NOT NULL',
            'DROP INDEX tmp_idx_1999dbe27c13 ON alerts ALGORITHM = INPLACE',
        ];

        $this->helper->updateColumnInBatches('alerts', 'message', '\'test message\'');

        $sql = $this->helper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }

        $statements = [
            'CREATE INDEX CONCURRENTLY tmp_idx_1999dbe27c13 ON alerts (message)',
            'UPDATE alerts SET message = \'test message\' WHERE message != \'test message\' OR \'test message\' IS NULL AND message IS NOT NULL LIMIT 1000',
            'UPDATE alerts SET message = \'test message\' WHERE message != \'test message\' OR \'test message\' IS NULL AND message IS NOT NULL',
            'DROP INDEX CONCURRENTLY tmp_idx_1999dbe27c13',
        ];

        $this->psqlHelper->updateColumnInBatches('alerts', 'message', '\'test message\'');

        $sql = $this->psqlHelper->getSql();
        foreach ($sql as $index => $query) {
            static::assertEquals($statements[$index], $query->getStatement());
        }
    }

    /**
     * @SuppressWarnings("PMD.UnusedLocalVariable")
     */
    protected function setUp(): void
    {
        $mockPlatform         = $this->getMockBuilder(MySQLPlatform::class)->getMock();
        $psqlPlatform         = $this->getMockBuilder(PostgreSQLPlatform::class)->getMock();
        $this->mockConnection = $this->getMockBuilder(Connection::class)->disableOriginalConstructor()->getMock();

        $mockPlatform->method('quoteIdentifier')
            ->willReturnArgument(0);
        $mockPlatform->method('getColumnDeclarationSQL')->willReturn('status VARCHAR(255)');

        $psqlPlatform->method('quoteIdentifier')
            ->willReturnArgument(0);
        $psqlPlatform->method('getColumnDeclarationSQL')->willReturn('status VARCHAR(255)');

        $this->helper     = new MockMigrationHelper($mockPlatform, $this->mockConnection);
        $this->psqlHelper = new MockMigrationHelper($psqlPlatform, $this->mockConnection);
    }
}
