<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\DeploymentBundle;

use Cyber\DeploymentBundle\CyberDeploymentBundle;
use PHPUnit\Framework\TestCase;
use Symfony\Component\DependencyInjection\ContainerBuilder;

/**
 * @internal
 *
 * @covers \Cyber\DeploymentBundle\CyberDeploymentBundle
 */
class CyberDeploymentTest extends TestCase
{
    /** @var ContainerBuilder */
    private $container;

    /**
     * @inheritDoc
     */
    protected function setUp(): void
    {
        $this->container = new ContainerBuilder();
        $this->container->setParameter('kernel.environment', 'test');
        $this->container->setParameter('kernel.build_dir', '.');
    }

    public function testLoggerTaggingServiceExists(): void
    {
        $extension = (new CyberDeploymentBundle())->getContainerExtension();
        static::assertNotNull($extension);
        $extension->load([
            'cyber_deployment' => [
                'log_tagging' => [
                    'tags' => [
                        'sometag' => 'some_value',
                    ],
                ],
            ],
        ], $this->container);

        /** @var bool $serviceExists stan thinkgs this is false */
        $serviceExists = $this->container->has('cyber_deployment.log_tagging_processor');
        static::assertTrue($serviceExists);

        // Check that the class alias is registered
        static::assertTrue($this->container->has(\Cyber\DeploymentBundle\LogTaggingProcessor::class));

        // Test that the parameter has been set correctly
        static::assertTrue($this->container->hasParameter('cyber.deployment.log.tags'));
        static::assertEquals(['sometag' => 'some_value'], $this->container->getParameter('cyber.deployment.log.tags'));
    }

    public function testFixtureParams(): void
    {
        $extension = (new CyberDeploymentBundle())->getContainerExtension();
        static::assertNotNull($extension);
        $extension->load([
            'cyber_deployment' => [
                'fixtures' => [
                    'connections'      => ['a'],
                    'managers'         => ['b'],
                    'migration_table'  => 'c',
                    'isolate_managers' => true,
                ],
            ],
        ], $this->container);
        $def = $this->container->getDefinition('cyber_deployment.config');
        static::assertTrue($def->hasMethodCall('setFixtureConnections'));
        static::assertTrue($def->hasMethodCall('setFixtureManagers'));
        static::assertTrue($def->hasMethodCall('setMigrationTable'));
        static::assertTrue($def->hasMethodCall('setIsolateManagers'));
    }

    public function testMigrationsConfig(): void
    {
        $extension = (new CyberDeploymentBundle())->getContainerExtension();
        static::assertNotNull($extension);
        $extension->load([
            'cyber_deployment' => [
                'migrations' => [
                    'dev'        => '/path/to/dev',
                    'prod'       => '/path/to/prod',
                    'categories' => ['category1', 'category2'],
                ],
            ],
        ], $this->container);
        $def = $this->container->getDefinition('cyber_deployment.config');
        static::assertTrue($def->hasMethodCall('setMigrationDevPath'));
        static::assertTrue($def->hasMethodCall('setMigrationProdPath'));
        static::assertTrue($def->hasMethodCall('setMigrationCategories'));
    }

    public function testNoDowntimeMigrationsConfig(): void
    {
        $extension = (new CyberDeploymentBundle())->getContainerExtension();
        static::assertNotNull($extension);
        $extension->load([
            'cyber_deployment' => [
                'migrations' => [
                    'dev'         => '/path/to/dev',
                    'prod'        => '/path/to/prod',
                    'no_downtime' => [
                        'enabled'     => true,
                        'pre_deploy'  => '/path/to/pre',
                        'post_deploy' => '/path/to/post',
                    ],
                ],
            ],
        ], $this->container);
        $def = $this->container->getDefinition('cyber_deployment.config');
        static::assertTrue($def->hasMethodCall('setPreDeployPath'));
        static::assertTrue($def->hasMethodCall('setPostDeployPath'));
    }

    public function testCommandServicesRegistration(): void
    {
        $extension = (new CyberDeploymentBundle())->getContainerExtension();
        static::assertNotNull($extension);
        $extension->load([
            'cyber_deployment' => [],
        ], $this->container);

        // Test command services from console.php
        static::assertTrue($this->container->has('cyber_deployment.command.fixtures'));
        static::assertTrue($this->container->has('cyber_deployment.command.merge_migrations'));
        static::assertTrue($this->container->has('cyber_deployment.command.validate'));
        static::assertTrue($this->container->has('cyber_deployment.command.no_downtime.execute'));
        static::assertTrue($this->container->has('cyber_deployment.command.no_downtime.update'));
        static::assertTrue($this->container->has('cyber_deployment.command.no_downtime.verify'));
    }

    public function testFixtureEventProducerServiceExists(): void
    {
        $extension = (new CyberDeploymentBundle())->getContainerExtension();
        static::assertNotNull($extension);
        $extension->load([
            'cyber_deployment' => [],
        ], $this->container);

        static::assertTrue($this->container->has('cyber_deployment.fixtures.event_producer'));
        $definition = $this->container->getDefinition('cyber_deployment.fixtures.event_producer');
        static::assertEquals(\Cyber\DeploymentBundle\Fixtures\FixtureEventProducer::class, $definition->getClass());
    }
}
