<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\FormExtrasBundle\Form\Type;

use Cyber\FormExtrasBundle\USStateList;
use InvalidArgumentException;
use Symfony\Component\Form\AbstractType;
use Symfony\Component\Form\ChoiceList\ArrayChoiceList;
use Symfony\Component\Form\ChoiceList\ChoiceListInterface;
use Symfony\Component\Form\ChoiceList\Loader\ChoiceLoaderInterface;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\OptionsResolver\Options;
use Symfony\Component\OptionsResolver\OptionsResolver;

class StateType extends AbstractType implements ChoiceLoaderInterface
{
    /**
     * State loaded choice list.
     *
     * Lazy loaded from USStateList component.
     *
     * @var ArrayChoiceList
     */
    private $choiceList;

    /**
     * @inheritdoc
     *
     * @noinspection ExceptionsAnnotatingAndHandlingInspection
     *
     * @SuppressWarnings(PHPMD.LongVariable)
     */
    public function configureOptions(OptionsResolver $resolver): void
    {
        $choicesNormalizer = function ($options, $value) {
            if (!empty($value)) {
                throw new InvalidArgumentException('Using "choices" option is not allowed for this type.');
            }

            return null;
        };

        $choiceLoaderNormalizer = function ($options, $value) {
            if (null !== $value) {
                throw new InvalidArgumentException('Overriding "choice_loader" option is not allowed for this type.');
            }

            return $this;
        };

        $choiceLabelNormalizer = function (Options $options, $choiceLabel) {
            if (null !== $choiceLabel) {
                //if overridden then return the set value
                return $choiceLabel;
            }

            return function ($choice, $key) use ($options) {
                $nameParts = [];
                if ($options['short_name']) {
                    $nameParts[] = $choice;
                }

                if ($options['long_name']) {
                    $nameParts[] = $key;
                }

                return \implode($options['separator'], $nameParts);
            };
        };

        $resolver
            ->setDefaults([
                'short_name' => true,
                'long_name'  => false,
                'separator'  => ' - ',
            ])
            ->setNormalizer('choices', $choicesNormalizer)
            ->setNormalizer('choice_loader', $choiceLoaderNormalizer)
            ->setNormalizer('choice_label', $choiceLabelNormalizer);
    }

    /**
     * @inheritdoc
     */
    public function getParent()
    {
        return ChoiceType::class;
    }

    /**
     * @inheritdoc
     *
     * @SuppressWarnings(PHPMD.StaticAccess)
     */
    public function loadChoiceList($value = null): ChoiceListInterface
    {
        if (null !== $this->choiceList) {
            return $this->choiceList;
        }

        return $this->choiceList = new ArrayChoiceList(USStateList::getStates(true), $value);
    }

    /**
     * @param string[]      $values
     * @param null|callable $value
     *
     * @return array<mixed>
     */
    public function loadChoicesForValues(array $values, $value = null): array
    {
        // If no callable is set, values are the same as choices
        if (null === $value) {
            return $values;
        }

        return $this->loadChoiceList($value)->getChoicesForValues($values);
    }

    /**
     * @param array<mixed>  $choices
     * @param null|callable $value
     *
     * @return string[]
     */
    public function loadValuesForChoices(array $choices, $value = null): array
    {
        // If no callable is set, choices are the same as values
        if (null === $value) {
            return $choices;
        }

        return $this->loadChoiceList($value)->getValuesForChoices($choices);
    }
}
