<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\FormExtrasBundle\Form\Extension;

use Doctrine\Common\Collections\ArrayCollection;
use LogicException;
use OpenApi\Annotations\Property;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractTypeExtension;
use Symfony\Component\Form\CallbackTransformer;
use Symfony\Component\Form\Event\PreSubmitEvent;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\OptionsResolver\Options;
use Symfony\Component\OptionsResolver\OptionsResolver;

class EntityTypeExtension extends AbstractTypeExtension
{
    /**
     * @inheritDoc
     */
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        // Due to change in symfony form component, we now have to force this to be multi select.
        $builder->addModelTransformer(new CallbackTransformer(function ($test) {
        }, function (ArrayCollection $collection) use ($options) {
            if ($options['cyber_multiple']) {
                return $collection;
            }

            return $collection->first() ?: null;
        }));

        $builder->addEventListener(FormEvents::PRE_SUBMIT, function (PreSubmitEvent $event) use ($options) {
            /** @var null|array{id?: scalar}|scalar $data */
            $data = $event->getData();
            if (null === $data) {
                return;
            }

            /** @var string $idField */
            $idField = $options['cyber_id_field'];
            if (!$options['cyber_multiple']) {
                $event->setData([$this->extractId($data, $idField)]);

                return;
            }

            \assert(\is_array($data), 'Data submitted for multiple select is not an array.');
            // multiple should be a nested array.
            foreach ($data as &$datum) {
                $datum = $this->extractId($datum, $idField);
            }
            $event->setData($data);
        }, 500);
    }

    /**
     * @inheritDoc
     */
    public function configureOptions(OptionsResolver $resolver): void
    {
        $wasMultiple = false;
        // we need to force multiple to be true internally, but save reference to original option.
        $multipleNormalizer = function (Options $options, $multiple) use (&$wasMultiple) {
            if ($multiple) {
                $wasMultiple = true;
            }

            return true;
        };

        $lazyCyberMultiple = function (Options $options) use (&$wasMultiple) {
            if ($options['multiple'] && $wasMultiple) {
                return true;
            }

            return false;
        };

        // from EntityType it is multiple so convert all values to arrays
        $emptyDataNormalizer = function (Options $options, $data) {
            // todo test multiple true and false
            if (null === $data) {
                return []; // no data = empty array because of multiple
            }

            return [$data];
        };

        $emptyData = function (Options $options) {
            if ($options['cyber_multiple']) {
                return [];
            }

            return null;
        };

        $docNormalizer = function (Options $options, $data) {
            return \array_merge([
                'type'        => 'object',
                'properties'  => [new Property(['property' => 'id', 'type' => 'integer'])],
                'description' => 'id itself or object with "id" property',
            ], $data ?: []);
        };

        $resolver->setNormalizer('multiple', $multipleNormalizer);

        // empty data now must always be normalized because we actually are not doing multiple selects
        $resolver->setNormalizer('empty_data', $emptyDataNormalizer);
        if ($resolver->isDefined('documentation')) {
            $resolver->setNormalizer('documentation', $docNormalizer);
        }

        // default empty data relies on 'multiple' flag so replace it with our own since we are not using it as intended
        $resolver->setDefault('empty_data', $emptyData);
        $resolver->setDefault('cyber_multiple', $lazyCyberMultiple);
        $resolver->setDefault('cyber_id_field', 'id');
        $resolver->setAllowedTypes('cyber_id_field', 'string');
    }

    /**
     * @inheritDoc
     */
    public static function getExtendedTypes(): iterable
    {
        return [EntityType::class];
    }

    /**
     * @param mixed[]|scalar $data
     */
    private function extractId($data, string $idField): mixed
    {
        if (!\is_array($data)) {
            // id was directly passed as value,
            return $data;
        }

        if (!isset($data[$idField])) {
            throw new LogicException(\sprintf('Value must be either an id or an object with %s property.', $idField));
        }

        return $data[$idField];
    }
}
