<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Tests\Cyber\FormExtrasBundle\Form\Extension;

use Cyber\FormExtrasBundle\Form\Extension\EntityTypeExtension;
use Doctrine\Common\Collections\ArrayCollection;
use Doctrine\ORM\EntityManagerInterface;
use Doctrine\ORM\Mapping\ClassMetadata;
use Doctrine\ORM\Query;
use Doctrine\ORM\Query\Expr;
use Doctrine\ORM\QueryBuilder;
use Doctrine\Persistence\ManagerRegistry;
use PHPUnit\Framework\MockObject\MockObject;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractExtension;
use Symfony\Component\Form\Test\TypeTestCase;

/**
 * @internal
 *
 * @coversNothing
 */
class EntityTypeExtensionTest extends TypeTestCase
{
    private MockObject&EntityManagerInterface $em;

    private QueryBuilder $queryBuilder;

    private MockObject&ManagerRegistry $emRegistry;

    private MockObject&Query $query;

    protected function setUp(): void
    {
        $this->em           = $this->createMock(EntityManagerInterface::class);
        $this->emRegistry   = $this->createMock(ManagerRegistry::class);
        $this->queryBuilder = $this->createMock(QueryBuilder::class);
        $this->query        = $this->createMock(Query::class);
        $classMeta          = $this->createMock(ClassMetadata::class);
        $this->em->method('getClassMetadata')->willReturn($classMeta);
        $this->em->method('contains')->willReturn(true);
        $this->em->method('createQuery')->willReturn($this->query);
        $classMeta->method('getName')->willReturn('FakeClass');
        $classMeta->method('getIdentifierFieldNames')->willReturn(['id']);
        $classMeta->method('getTypeOfField')->willReturn('integer');
        $classMeta->method('getIdentifierValues')->willReturnCallback(fn ($obj) => [$obj->getId()]);

        $this->emRegistry->method('getManagerForClass')->with('FakeClass')->willReturn($this->em);
        $this->queryBuilder->method('getRootAliases')->willReturn(['root']);
        $this->queryBuilder->method('getRootEntities')->willReturn(['FakeClass']);
        $this->queryBuilder->method('getEntityManager')->willReturn($this->em);
        $this->queryBuilder->method('getParameters')->willReturn(new ArrayCollection());
        $this->queryBuilder->method('getQuery')->willReturn($this->query);
        $this->queryBuilder->method('andWhere')->willReturnSelf();
        $this->queryBuilder->method('expr')->willReturn(new Expr());

        $this->query->method('setParameter')->with(static::anything(), ['1'])->willReturnSelf();
        parent::setUp();
    }

    public function testSubmitValidData(): void
    {
        $entity = new class () implements MockEntityInterface {
            public function getId(): string
            {
                return '1';
            }
        };
        $this->query->method('getResult')->willReturn([$entity]);

        $formData = ['id' => '1'];

        // $model will retrieve data from the form submission; pass it as the second argument
        $form = $this->factory->create(EntityType::class, null, [
            'class'         => 'FakeClass',
            'query_builder' => $this->queryBuilder,
        ]);

        $form->submit($formData);
        static::assertTrue($form->isSynchronized(), (string) $form->getTransformationFailure());
        static::assertTrue($form->isValid());
        $data = $form->getData();
        static::assertSame($entity, $data);

        $formData = '1';

        // $model will retrieve data from the form submission; pass it as the second argument
        $form = $this->factory->create(EntityType::class, null, [
            'class'         => 'FakeClass',
            'query_builder' => $this->queryBuilder,
        ]);

        $form->submit($formData);
        static::assertTrue($form->isSynchronized(), (string) $form->getTransformationFailure());
        static::assertTrue($form->isValid());
        $data = $form->getData();
        static::assertSame($entity, $data);
    }

    public function testSubmitMultiple(): void
    {
        $this->query->method('getResult')->willReturn([new class () implements MockEntityInterface {
            public function getId(): string
            {
                return '1';
            }
        }]);

        $formData = [['id' => '1']];

        // $model will retrieve data from the form submission; pass it as the second argument
        $form = $this->factory->create(EntityType::class, null, [
            'class'         => 'FakeClass',
            'multiple'      => true,
            'query_builder' => $this->queryBuilder,
        ]);

        $form->submit($formData);
        static::assertTrue($form->isSynchronized(), (string) $form->getTransformationFailure());
        static::assertTrue($form->isValid());
        /**
         * @var mixed[] $data
         */
        $data = $form->getData();
        static::assertIsIterable($data);
        static::assertCount(1, $data);
        static::assertInstanceOf(MockEntityInterface::class, $data[0]);
        static::assertEquals('1', $data[0]->getId());

        $formData = ['1'];

        // $model will retrieve data from the form submission; pass it as the second argument
        $form = $this->factory->create(EntityType::class, null, [
            'class'         => 'FakeClass',
            'multiple'      => true,
            'query_builder' => $this->queryBuilder,
        ]);

        $form->submit($formData);
        static::assertTrue($form->isSynchronized(), (string) $form->getTransformationFailure());
        static::assertTrue($form->isValid());
        /**
         * @var mixed[] $data
         */
        $data = $form->getData();
        static::assertIsIterable($data);
        static::assertCount(1, $data);
        static::assertInstanceOf(MockEntityInterface::class, $data[0]);
        static::assertEquals('1', $data[0]->getId());
    }

    /**
     * @return AbstractExtension[]
     */
    protected function getExtensions(): array
    {
        return [
            new class ($this->emRegistry) extends AbstractExtension {
                private ManagerRegistry $registry;

                public function __construct(ManagerRegistry $registry)
                {
                    $this->registry = $registry;
                }

                protected function loadTypes(): array
                {
                    return [new EntityType($this->registry)];
                }

                protected function loadTypeExtensions(): array
                {
                    return [new EntityTypeExtension()];
                }
            },
        ];
    }
}
