<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\FormExtrasBundle\Form\Extension;

use Doctrine\Common\Collections\ArrayCollection;
use LogicException;
use OpenApi\Annotations\Property;
use OpenApi\Annotations\Schema;
use Symfony\Bridge\Doctrine\Form\Type\EntityType;
use Symfony\Component\Form\AbstractTypeExtension;
use Symfony\Component\Form\CallbackTransformer;
use Symfony\Component\Form\Event\PreSubmitEvent;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormEvents;
use Symfony\Component\OptionsResolver\Options;
use Symfony\Component\OptionsResolver\OptionsResolver;

class EntityTypeExtension extends AbstractTypeExtension
{
    /**
     * @inheritDoc
     */
    public function buildForm(FormBuilderInterface $builder, array $options): void
    {
        // Due to change in symfony form component, we now have to force this to be multi select.
        $builder->addModelTransformer(new CallbackTransformer(function ($test) {
        }, function (ArrayCollection $collection) use ($options) {
            if ($options['cyber_multiple']) {
                return $collection;
            }

            return $collection->first() ?: null;
        }));

        $builder->addEventListener(FormEvents::PRE_SUBMIT, function (PreSubmitEvent $event) use ($options) {
            /** @var null|array{id?: scalar}|scalar $data */
            $data = $event->getData();
            if (null === $data) {
                return;
            }

            /** @var string $idField */
            $idField = $options['cyber_id_field'];
            if (!$options['cyber_multiple']) {
                $event->setData([$this->extractId($data, $idField)]);

                return;
            }

            \assert(\is_array($data), 'Data submitted for multiple select is not an array.');
            // multiple should be a nested array.
            foreach ($data as &$datum) {
                $datum = $this->extractId($datum, $idField);
            }
            unset($datum);
            $event->setData($data);
        }, 500);
    }

    /**
     * @inheritDoc
     */
    public function configureOptions(OptionsResolver $resolver): void
    {
        $wasMultiple = false;
        // we need to force multiple to be true internally, but save reference to original option.
        $multipleNormalizer = function (Options $options, $multiple) use (&$wasMultiple) {
            // since forms are services the configureOptions is called only once per lifetime or request.
            // as a result if a form uses multiple EntityTypes the referenced var $wasMultiple
            // will be initialized above only once, and then has to be reset to proper state by each call to normalizer.
            $wasMultiple = $multiple;

            return true;
        };

        $lazyCyberMultiple = function (Options $options) use (&$wasMultiple) {
            return $options['multiple'] && $wasMultiple;
        };

        // from EntityType it is multiple so convert all values to arrays
        $emptyDataNormalizer = function (Options $options, $data) use (&$wasMultiple) {
            if (null === $data || ($options['multiple'] && $wasMultiple && empty($data))) {
                return []; // no data = empty array because of multiple
            }

            return [$data];
        };

        $emptyData = function (Options $options) {
            if ($options['cyber_multiple']) {
                return [];
            }

            return null;
        };

        $docNormalizer = function (Options $options, $data) {
            return $this->nelmioDocNormalizer($options, $data);
        };

        $resolver->setNormalizer('multiple', $multipleNormalizer);

        // empty data now must always be normalized because we actually are not doing multiple selects
        $resolver->setNormalizer('empty_data', $emptyDataNormalizer);
        if ($resolver->isDefined('documentation')) {
            $resolver->setNormalizer('documentation', $docNormalizer);
        }

        // default empty data relies on 'multiple' flag so replace it with our own since we are not using it as intended
        $resolver->setDefault('empty_data', $emptyData);
        $resolver->setDefault('cyber_multiple', $lazyCyberMultiple);
        $resolver->setDefault('cyber_id_field', 'id');
        $resolver->setAllowedTypes('cyber_id_field', 'string');
        // used only for nelmio documentation generation
        $resolver->setDefault('cyber_id_type', 'integer');
        $resolver->setAllowedTypes('cyber_id_field', 'string');
    }

    /**
     * @inheritDoc
     */
    public static function getExtendedTypes(): iterable
    {
        return [EntityType::class];
    }

    /**
     * @param mixed[]|scalar $data
     */
    private function extractId($data, string $idField): mixed
    {
        if (!\is_array($data)) {
            // id was directly passed as value,
            return $data;
        }

        if (!isset($data[$idField])) {
            throw new LogicException(\sprintf('Value must be either an id or an object with %s property.', $idField));
        }

        return $data[$idField];
    }

    /**
     * Adds a better documentation for EntityTypes.
     *
     * Since our extension is doing magic with 'multiple' Nelmio is not able to guess types correctly.
     *
     * @param Options      $options
     * @param null|mixed[] $data
     *
     * @return mixed
     */
    public function nelmioDocNormalizer(Options $options, $data): mixed
    {
        if (!\class_exists(Schema::class)) {
            // some other lib added documentation key or we are outdated, just return data.
            return $data;
        }
        /** @var string $idField */
        $idField = $options['cyber_id_field'];
        /** @var string $idType */
        $idType = $options['cyber_id_type'];

        $example = match ($idType) {
            'string'  => 'abc123',
            'integer' => 123,
            'uuid'    => '01978b4b-71f5-7e71-a6ea-73fe27fd74aa',
            default   => $idType,
        };

        if ('uuid' === $idType) {
            $idType = 'string';
        }

        $itemDefinition = [
            'oneOf'       => [
                new Schema(
                    [
                        'type'        => 'object',
                        'properties'  => [
                            new Property([
                                'property'    => $idField,
                                'type'        => $idType,
                                'description' => 'Id of target entity',
                                'example'     => $example,
                            ]),
                        ],
                        'description' => 'Object containing id of target entity. All other props are ignored',
                    ]
                ),
                new Schema([
                    'type'        => $idType,
                    'description' => 'Id of target entity',
                    'example'     => $example,
                ]),
            ],
            'description' => \sprintf('id itself or any object with "%s" property', $idField),
        ];

        if ($options['cyber_multiple']) {
            return \array_merge([
                'type'  => 'array',
                'items' => $itemDefinition,
            ], $data ?: []);
        }

        $itemDefinition['type']    = 'object';
        $itemDefinition['example'] = \sprintf('%s or {%s: %s}', $example, $idField, $example);

        return \array_merge(
            $itemDefinition,
            $data ?: []
        );
    }
}
