<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\IdpBundle\Controller;

use Cyber\IdpBundle\Exception\SamlProcessingException;
use Cyber\IdpBundle\Service\AuthnRequestProcessor;
use Cyber\IdpBundle\Service\SamlConfig;
use Cyber\IdpBundle\Service\SamlResponseGenerator;
use LightSaml\Model\Context\DeserializationContext;
use LightSaml\Model\Context\SerializationContext;
use LightSaml\Model\Metadata;
use LightSaml\Model\Protocol;
use LightSaml\SamlConstants;
use Symfony\Bundle\FrameworkBundle\Controller\AbstractController;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Routing\Attribute\Route;
use Symfony\Component\Routing\Generator\UrlGeneratorInterface;

/**
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects) TODO refactor
 */
class SamlIdpController extends AbstractController
{
    public function __construct(
        private SamlConfig $samlConfig,
        private AuthnRequestProcessor $authnProcessor,
        private SamlResponseGenerator $responseGenerator,
    ) {
    }

    #[Route('/meta', methods: ['GET'])]
    public function metadata(): Response
    {
        // Entity Descriptor for the IdP
        $entityDescriptor = new Metadata\EntityDescriptor();
        $entityDescriptor->setEntityID($this->samlConfig->entityId);

        // IdP SSO Descriptor
        $idpSsoDescriptor = new Metadata\IdpSsoDescriptor();
        $idpSsoDescriptor->setProtocolSupportEnumeration(SamlConstants::PROTOCOL_SAML2);

        // SSO Endpoint (SingleSignOnService)
        $singleSignOnService = new Metadata\SingleSignOnService();
        $singleSignOnService
            ->setBinding(SamlConstants::BINDING_SAML2_HTTP_REDIRECT)
            ->setLocation($this->generateUrl('cyber_idp_saml_sso', [], UrlGeneratorInterface::ABSOLUTE_URL));
        $idpSsoDescriptor->addSingleSignOnService($singleSignOnService);

        // Add Certificate (KeyDescriptor)
        $cert          = $this->responseGenerator->getCert();
        $keyDescriptor = new Metadata\KeyDescriptor();
        $keyDescriptor->setUse(Metadata\KeyDescriptor::USE_SIGNING)
            ->setCertificate($cert);
        $idpSsoDescriptor->addKeyDescriptor($keyDescriptor);

        // Set the IdP SSO Descriptor in the Entity Descriptor
        $entityDescriptor->addItem($idpSsoDescriptor);

        // Create the serialization context
        $serializationContext = new SerializationContext();
        // Serialize the EntityDescriptor to XML
        $entityDescriptor->serialize($serializationContext->getDocument(), $serializationContext);

        // Get the XML as a string
        $xml = $serializationContext->getDocument()->saveXML() ?: 'Could not encode XML.';

        // Return the response with proper headers
        $response = new Response($xml);
        $response->headers->set('Content-Type', 'application/xml');

        return $response;
    }

    #[Route('/sso', methods: ['GET'], name: 'cyber_idp_saml_sso')]
    public function sso(Request $request): Response
    {
        // Parse the SAML AuthnRequest from the Service Provider (SP)
        $samlRequest = $request->query->get('SAMLRequest', '');

        // Decode the request (HTTP Redirect binding involves base64 and unzipping)
        $samlRequestXml = \gzinflate(\base64_decode($samlRequest, true) ?: '');

        try {
            if (!$samlRequestXml) {
                throw new SamlProcessingException('Could not decode SAMLRequest');
            }
            // Create deserialization context and load the SAML request
            $desContext = new DeserializationContext();
            $desContext->getDocument()->loadXML($samlRequestXml);

            $authnRequestNode = $desContext->getDocument()->firstChild;
            if (!$authnRequestNode) {
                throw new SamlProcessingException('Could not decode AuthnRequest');
            }

            // Create the AuthnRequest object from the deserialized XML
            $authnRequest = new Protocol\AuthnRequest();
            $authnRequest->deserialize($authnRequestNode, $desContext);

            $response = $this->authnProcessor->process($authnRequest, $request);

            // Serialize the response to XML
            $serializationContext = new SerializationContext();
            $response->serialize($serializationContext->getDocument(), $serializationContext);

            $respXml = $serializationContext->getDocument()->saveXML();
            if (!$respXml) {
                throw new SamlProcessingException('Could not generate response XML');
            }
            // Base64 encode the response for HTTP-POST binding
            $samlResponse = \base64_encode($respXml);

            $relayState = \htmlspecialchars($request->query->get('RelayState', ''), ENT_QUOTES | ENT_HTML5);

            // Create a form to auto-submit the SAMLResponse
            $acsUrl = $authnRequest->getAssertionConsumerServiceURL();

            $html = "
            <html>
            <body>
                <form method='POST' action='{$acsUrl}' id='saml-form'>
                    <input type='hidden' name='SAMLResponse' value='{$samlResponse}' />
                    <input type='hidden' name='RelayState' value='{$relayState}' />
                </form>
                <script>document.getElementById('saml-form').submit();</script>
            </body>
            </html>
        ";

            return new Response($html);
        } catch (SamlProcessingException $ex) {
            // todo exception class
            return new Response($ex->getMessage(), 400);
        }
    }
}
