<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Cyber\IdpBundle\Service;

use Cyber\IdpBundle\Entity\ServiceProvider;
use Cyber\IdpBundle\Exception\SamlProcessingException;
use DateTime;
use LightSaml\Credential\KeyHelper;
use LightSaml\Credential\X509Certificate;
use LightSaml\Error\LightSamlSecurityException;
use LightSaml\Model\Assertion;
use LightSaml\Model\Protocol;
use LightSaml\Model\XmlDSig\SignatureStringReader;
use LightSaml\SamlConstants;
use Symfony\Component\HttpFoundation\Request;

/**
 * This class is responsible for processing SAML Authentication Requests (AuthnRequest).
 * It validates the incoming request, ensures the request is correctly signed by the service provider (SP),
 * and generates a SAML response to return to the asserting party.
 *
 * @SuppressWarnings(PHPMD.StaticAccess) saml uses a lot of statics
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects) TODO refactor
 */
class AuthnRequestProcessor
{
    public function __construct(
        private readonly SamlConfig $samlConfig,
        private readonly SamlResponseGenerator $responseGenerator,
    ) {
    }

    public function process(Protocol\AuthnRequest $authnRequest, Request $request): Protocol\Response
    {
        // before anything we should validate the request
        $issuer = $authnRequest->getIssuer()?->getValue();
        if (!$issuer) {
            throw new SamlProcessingException('No valid issuer in the request');
        }
        $svProvider = $this->samlConfig->spRepo->findByEntityId($issuer);
        if (!$svProvider) {
            throw new SamlProcessingException('Unsupported Issuer');
        }

        // we can only validate it after we've retrieved the SP, but for validation we need original request
        if (!$this->verifyRequest($svProvider, $request)) {
            throw new SamlProcessingException('Invalid signature');
        }

        return $this->responseGenerator->generate(
            $svProvider,
            function (Protocol\Response $response) use ($authnRequest, $issuer) {
                $assertion = $response->getFirstAssertion();
                \assert(null !== $assertion, 'Response was expected to have at least one assertion');
                $subject = $assertion->getSubject();
                \assert(null !== $subject, 'Assertion was expected to have subject');
                $conditions = $assertion->getConditions();
                \assert(null !== $conditions, 'Assertion was expected to have conditions');

                // Subject confirmation
                $subjectConfirmation = new Assertion\SubjectConfirmation();
                $subjectConfirmation->setMethod(SamlConstants::CONFIRMATION_METHOD_BEARER);
                $subConfirmationData = new Assertion\SubjectConfirmationData();
                $subConfirmationData->setInResponseTo($authnRequest->getID());
                $subConfirmationData->setRecipient($authnRequest->getAssertionConsumerServiceURL());
                $subConfirmationData->setNotOnOrAfter(new DateTime('+5 minutes'));
                $subjectConfirmation->setSubjectConfirmationData($subConfirmationData);
                $subject->addSubjectConfirmation($subjectConfirmation);

                // Create AudienceRestriction
                $audienceRestriction = new Assertion\AudienceRestriction($issuer);

                // Add AudienceRestriction to Conditions
                $conditions->addItem($audienceRestriction);

                // Add consumer data to response
                $response->setDestination($authnRequest->getAssertionConsumerServiceURL());
                $response->setInResponseTo($authnRequest->getID());
            }
        );
    }

    private function verifyRequest(ServiceProvider $svProvider, Request $request): bool
    {
        $cert = new X509Certificate();
        $cert->loadPem($svProvider->getCertificate());

        // Get the public key from the certificate
        $publicKey = KeyHelper::createPublicKey($cert);

        // Extract parameters
        $samlRequest = $request->query->get('SAMLRequest', '');
        $relayState  = $request->query->get('RelayState', '');
        $sigAlg      = $request->query->get('SigAlg', '');
        $signature   = $request->query->get('Signature', '');

        // Rebuild the signed payload
        $signedPayload = 'SAMLRequest=' . \urlencode($samlRequest);
        if ($relayState) {
            $signedPayload .= '&RelayState=' . \urlencode($relayState);
        }
        $signedPayload .= '&SigAlg=' . \urlencode($sigAlg);

        // Use SignatureStringReader to validate the signature
        $signatureReader = new SignatureStringReader($signature, $sigAlg, $signedPayload);

        try {
            // Validate the signature
            $signatureReader->validate($publicKey);

            return true;
        } catch (LightSamlSecurityException $e) {
            return false;
        }
    }
}
