<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\IdpBundle\Service;

use Cyber\IdpBundle\Entity\ServiceProvider;
use DateTime;
use LightSaml\Credential\KeyHelper;
use LightSaml\Credential\X509Certificate;
use LightSaml\Helper;
use LightSaml\Model\Assertion;
use LightSaml\Model\Protocol;
use LightSaml\Model\XmlDSig\SignatureWriter;
use LightSaml\SamlConstants;
use RobRichards\XMLSecLibs\XMLSecurityKey;

/**
 * Class responsible for generating SAML responses for Service Providers (SP).
 *
 * @SuppressWarnings(PHPMD.CouplingBetweenObjects) TODO refactor
 * @SuppressWarnings(PHPMD.StaticAccess) saml uses a lot of statics
 */
class SamlResponseGenerator
{
    public function __construct(
        private readonly SamlConfig $samlConfig,
    ) {
    }

    public function generate(ServiceProvider $svProvider, callable $finalizeCb): Protocol\Response
    {
        // before creating response, collect all require assertion data;
        $assertionProvider = $this->samlConfig->spRepo->getAssertionProvider($svProvider);
        $userId            = $assertionProvider->getUserNameId($svProvider);

        // Create an AttributeStatement
        $attributeStatement = new Assertion\AttributeStatement();

        // add all attributes to the statement
        foreach ($assertionProvider->getUserAttributes($svProvider) as $name => $value) {
            $assertAttribute = new Assertion\Attribute($name, $value);
            $attributeStatement->addAttribute($assertAttribute);
        }

        // Generate a SAML Response
        $response = new Protocol\Response();
        $response->setID(Helper::generateID());
        $response->setIssueInstant(new DateTime());
        $response->setIssuer(new Assertion\Issuer($this->samlConfig->entityId));

        // Add status (Success)
        $status = new Protocol\Status(new Protocol\StatusCode(SamlConstants::STATUS_SUCCESS));
        $response->setStatus($status);

        // Build the assertion
        $assertion = new Assertion\Assertion();
        $assertion->setID(Helper::generateID());
        $assertion->setIssueInstant(new DateTime());
        $assertion->setIssuer(new Assertion\Issuer($this->samlConfig->entityId));

        // Add the subject (user identifier)
        $subject = new Assertion\Subject();
        $nameID  = new Assertion\NameID($userId, SamlConstants::NAME_ID_FORMAT_PERSISTENT);
        $subject->setNameID($nameID);

        $assertion->setSubject($subject);

        // Add conditions
        $conditions = new Assertion\Conditions();
        $conditions->setNotBefore(new DateTime());
        $conditions->setNotOnOrAfter(new DateTime('+5 minutes'));
        $assertion->setConditions($conditions);

        // Add authentication statement
        $authnStatement = new Assertion\AuthnStatement();
        $authnStatement->setAuthnInstant(new DateTime());

        // Create the AuthnContext object
        $authnContext = new Assertion\AuthnContext();
        $authnContext->setAuthnContextDecl(SamlConstants::AUTHN_CONTEXT_PASSWORD_PROTECTED_TRANSPORT);

        // Add it to the AuthnStatement
        $authnStatement->setAuthnContext($authnContext);

        $assertion->addItem($authnStatement);

        // Add the AttributeStatement to the assertion
        $assertion->addItem($attributeStatement);

        // Create Conditions
        $conditions = new Assertion\Conditions();
        $conditions->setNotBefore(new DateTime('-1 minute'));
        $conditions->setNotOnOrAfter(new DateTime('+10 minutes')); // Set a valid time range

        // Add Conditions to Assertion
        $assertion->setConditions($conditions);

        // Add the assertion to the response
        $response->addAssertion($assertion);

        $finalizeCb($response);

        // Create key and certificate objects
        $privateKey  = KeyHelper::createPrivateKey($this->samlConfig->key, '', false, XMLSecurityKey::RSA_SHA256);
        $certificate = $this->getCert();
        $writer      = new SignatureWriter($certificate, $privateKey);
        $assertion->setSignature($writer);
        $response->setSignature($writer);

        return $response;
    }

    public function getCert(): X509Certificate
    {
        $cert = new X509Certificate();
        $cert->loadPem($this->samlConfig->certificate);

        return $cert;
    }
}
