<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\IdpBundle\Service;

use Cyber\IdpBundle\Event\FinalizeAssertionsEvent;
use Cyber\IdpBundle\Event\FinalizeResponseEvent;
use Cyber\IdpBundle\Exception\SamlProcessingException;
use LightSaml\Model\Context\SerializationContext;
use LightSaml\Model\Protocol;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;

class SsoInitiator
{
    public function __construct(
        private readonly SamlConfig $samlConfig,
        private readonly SamlResponseGenerator $responseGenerator,
        private readonly EventDispatcherInterface $dispatcher,
    ) {
    }

    /**
     * Prepares and returns a response needed for IDP initiated SSO.
     *
     * @return string SAML Response XML string
     */
    public function generateIdpInitiatedSso(string $spIdentifier): string
    {
        $svProvider = $this->samlConfig->spRepo->findByEntityId($spIdentifier);
        if (!$svProvider) {
            throw new SamlProcessingException('Could not SP provider with identifier: ' . $spIdentifier);
        }

        $response = $this->responseGenerator->generate(
            $svProvider,
            function (Protocol\Response $response) use ($svProvider) {
                $assertion = $response->getFirstAssertion();
                \assert(null !== $assertion, 'Response was expected to have at least one assertion');

                $this->dispatcher->dispatch(new FinalizeAssertionsEvent($assertion, $svProvider));
                $this->dispatcher->dispatch(new FinalizeResponseEvent($response, $svProvider));
            }
        );

        // Serialize the response to XML
        $serializationContext = new SerializationContext();
        $response->serialize($serializationContext->getDocument(), $serializationContext);

        $respXml = $serializationContext->getDocument()->saveXML();
        if (!$respXml) {
            throw new SamlProcessingException('Could not generate response XML');
        }

        return $respXml;
    }
}
