<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

declare(strict_types=1);

namespace Tests\Cyber\IdpBundle\Service;

use Cyber\IdpBundle\Entity\ServiceProvider;
use Cyber\IdpBundle\Entity\ServiceProviderRepoInterface;
use Cyber\IdpBundle\Service\AssertionProviderInterface;
use Cyber\IdpBundle\Service\AuthnRequestProcessor;
use Cyber\IdpBundle\Service\SamlConfig;
use Cyber\IdpBundle\Service\SamlResponseGenerator;
use LightSaml\Model\Context\DeserializationContext;
use LightSaml\Model\Protocol\AuthnRequest;
use PHPUnit\Framework\TestCase;
use Symfony\Component\HttpFoundation\Request;

/**
 * @internal
 *
 * @coversNothing
 */
class AuthnRequestProcessorTest extends TestCase
{
    public function testProcess(): void
    {
        $request = 'SAMLRequest=lZJLa9tAEMfv%2FRRi76unJTuLpWBiAoY0lDwK7SUM0jjestpVd0Zp00%2FflWyH5hLodfk%2FZn8z68vfvYle0JN2thZZnIoIbes6bZ9r8fhwLVfisvm0JuhNPqjNyAd7hz9HJI42ROg5%2BK6cpbFHf4%2F%2BRbf4eHdTiwPzQCpJYE8cxDEEZzySRCCWWQw9%2FHGh59lqdnHr%2BmRuSHQ3eKQhBKKItsGoLfA82hQY8oxrwRwcsVqlizSBoU9OKXNCQuREtNvW4qmsuipLS5AlLCu5WBR7eZGVrcQCqgowxSUWQUo04s4Sg%2BVa5Gm%2BkOmFzFcPaa6KTKXLuCiX30X09YwonxAFaJbUEUotRm%2BVA9KkLPRIilt1v%2Fl8o4JUDd6xa50RzZGhmgt9dO18D%2Fyxd3rRndzPUoWWNb%2B%2B6%2F7YDuf9iGaSHZGrEy1Fg3pbxxPejt%2B6H9mqWyf%2FTtmc934bwnfbL87o9jXaGON%2BXXkExlrswUyr%2Bv%2FvDBPQcBuWRdKcat8fWPMX&RelayState=H4sIAAAAAAAAAD2QWXPaMBRG_4ueEdjW4uWNlACNU2oCoSSdjke2JG-yBVg2DJ3-94iXvp0798xdvr-AgQgMPRSsN9BNxWb44LUbcDABme0Y0RuLuUUSKM3PiLiFy3EwdqPHlAkLXpZW4FYojTn10WymdM5UqXsTYc9xZ0M1Y4N5SMJKuebCorS4m_94tViA6DcQLauULfRJdNVj-emiZaUE-DMBjXXjuV_spFeNC5L03lO_c8KUonMQoOLQJEET1svx-PL6Ee6d9V7UbKm-rTJyG543NxfCt21zX9brvF53P9dlF8tVckvaCzPFr5qNWwjl52G-bePF_n3RJQU9XTfJ6nDKuobEiDht_HK7q_OYDem1lMFwyQgP0fm5UXXPP9OjfjruvmuDi07W26s9XtmLmewf4U0fv0__BzxlLbvrLtdFVxk9zXVr9RZEru_5OHQo9e3rIJJM9WICLnYOFzTj1MEwRLkPMfUDmBGGYIgdF-FcMI8jO8NYNSWUU9chDBLmU4gxkjB0SQ4FYpQy4QhfIPDvC__u5gL0AQAA.H4sIAAAAAAAAALs6qTeW30-nYLvWwgNz3yjqGsWceHYvoFXQx0dd2fpoejwA41lZvCAAAAA.4&SigAlg=http%3A%2F%2Fwww.w3.org%2F2001%2F04%2Fxmldsig-more%23rsa-sha256&Signature=M5L3dFQ%2BjD%2FZSiO5Fzl9BZcDMp8qlTImU1RtjV9Ug5S7oeGjXXG%2BwNTRvQ%2BdXIasUnnAvf8GFIVerDNTM7y%2BknlMUZJFxVJz%2FfieNBj%2BDod0sZL%2FHyWJIzy42YkprBFm6rfIZmw4sfVVgDfKEnw62lcSyigduAOO1iYrgktdlkX4zA1Mfmgrx2%2Fehp%2F3%2FHELOYFM9mJgDeyeqkFMdWwd0VNqDlepjBeGlQz7R9DQcX7f0nNr1AYuCnlOgFhLF7qhjWW9GxMy2pz1QKxwlM8LMRZ4k9eDEUrob2%2B7oUk0cWNfRYARHinchBlfoHxSexPbafIDsAhPv2dTZ%2FEmHkHrAQ%3D%3D';
        \parse_str($request, $data);

        $provider = $this->createMock(ServiceProvider::class);
        $provider->method('getCertificate')->willReturn('-----BEGIN CERTIFICATE-----
MIICvTCCAaWgAwIBAgIJALA1f6+lcdeiMA0GCSqGSIb3DQEBCwUAMB4xHDAaBgNVBAMME3VzLWVhc3QtMV9lTnVZZGoxOGQwHhcNMjQwOTI1MDI0NDU2WhcNMzQwOTI1MTI1NjU2WjAeMRwwGgYDVQQDDBN1cy1lYXN0LTFfZU51WWRqMThkMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAuw8ABTQlYYRm6+LXv8mJ5QZny/xbz0NE/Za27vdN7pOIcaN5cpopzmzZUB2wOvzu/5n5DyOmkSvbS9OcqVxpp80pl241ZyovM3TNKab0t3haIS69PMNw24Nl5nEjj4vxKJ3HlKoMR78VzuyYajZ6iXiNZfiQRjL9syQ4ZR7MPrudAyG5Wgcw2t1b/ApsmC1/Cne5jq/v5SUO9KcdV77cHum9fwXcGnFjhoMPt3ioQJnmLzQDal+Xit5bH1GX3KEoqzUT1vpKOSTJMJ/ftSfyeQCkvH7zB4gGMg+LNMmm9vIb3jnNKrFrst1kdQ9i29vEsI6tJi2M00vMJqyuL9NoswIDAQABMA0GCSqGSIb3DQEBCwUAA4IBAQAsiOXl8LJqbths83nG48DIOiU1BekdoPZMz5W0TJTQuTqPL8TFQVJrHP1rTMp9w+IsQxmVUaYq0UIURbajxGIt83NjVElMVYasxFfwYQLAlz0LS/mh141AwnEMyj20iGUqe6iKfTt7ZSaNNAoBMUkeSoz75WCoBEHagwUCqMwDHDINgvZOzZoy1H9YEGymV+Q6ozIxakJy+3otcytidIHJ2RaAZUO2NlV4ykGdsja6SO2s8AmO5PPwquLa/Ur6LHCrijKFErTduvTa1R4kgU1A9TebVfXqc3ssgE9RnWfIchKJg1FL/QUOm39UHMhAwSjPqICVUpZsEwlUEnImvs+d
-----END CERTIFICATE-----');

        $assertionProvider = $this->createMock(AssertionProviderInterface::class);
        $assertionProvider->method('getUserNameId')->with($provider)->willReturn('xyz');

        $providerRepo = $this->createMock(ServiceProviderRepoInterface::class);
        $providerRepo->method('findByEntityId')->with('urn:amazon:cognito:sp:us-east-1_eNuYdj18d')
            ->willReturn($provider);
        $providerRepo->method('getAssertionProvider')->with($provider)->willReturn($assertionProvider);

        $request = new Request($data);
        $configs = new SamlConfig(
            'abc',
            $providerRepo,
            '-----BEGIN CERTIFICATE-----
MIIDRTCCAi0CFGGhINyavdr3Va9VHRz9T2RI3ZTFMA0GCSqGSIb3DQEBCwUAMF4xCzAJBgNVBAYTAlVTMQswCQYDVQQIDAJQQTEPMA0GA1UEBwwGRWFzdG9uMSEwHwYDVQQKDBhJbnRlcm5ldCBXaWRnaXRzIFB0eSBMdGQxDjAMBgNVBAMMBWN5YmVyMCAXDTI0MTAwMTA0MDY1MloYDzIwNTIwMjE2MDQwNjUyWjBeMQswCQYDVQQGEwJVUzELMAkGA1UECAwCUEExDzANBgNVBAcMBkVhc3RvbjEhMB8GA1UECgwYSW50ZXJuZXQgV2lkZ2l0cyBQdHkgTHRkMQ4wDAYDVQQDDAVjeWJlcjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBANmlY+IuiHc2+0Jfq7gjJ65mdcQmXq4aEG3mUoeK0dLFqR9WZAPUl9MmIZuy6C79JAY95sDUh1jKwN/tV8/XUAepBXsYsO9oJqo2heOVGYlhv4wm7YVe1vHPP1GwS/tKzEozIfvSeCGIErI7cpv1p9R/+ntQ88DsP/zrIji7WWx4CHqPr4NJ8Ez/wW3MIMRquSKfRKTpEABT51y6pvxPDiVCpuPKwN1R42yXipPASEg8K3EIG1nsn/TqnNWSIM4f3Mph2NWyA2R3hsFDTTY1MhksZ1mVJg2kyagDQ/iNXi5y7dQTcwff648839h/DTtnHd3/vGCr7VJ/oVVO0XzuPfcCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEASkQIA/K9gkySPY0bOxaCmeyVV66RGojjTQhJ8oVY6q8pY5ICdGDw8erX3Sbj7dhGb8zV2zeuk2BPHHDYdnDwStNL0tz05A73czVkXW6SI2AHdMlUgTayRed3PoVAQSukS363fVeto2HRokrSkmvYwkQB45ds74t6QwrbPOkPiLhlOpeqL4wvYLSxdWjpow9j1MNZwJD7aWLr58qE0ASDFz+kWsIdryYh9HwfY0brv1VX8VeFj7JVvYf+Hq72ouV8GlkQKRLbPYKdD5GUcBIkZLCiGfWoiJFqq+gvsF7t3FMcefRzPIWb4c/6MJDvnAoKDxYHthtAu9vG2mK8ioESVQ==
-----END CERTIFICATE-----',
            '-----BEGIN PRIVATE KEY-----
MIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDZpWPiLoh3NvtCX6u4IyeuZnXEJl6uGhBt5lKHitHSxakfVmQD1JfTJiGbsugu/SQGPebA1IdYysDf7VfP11AHqQV7GLDvaCaqNoXjlRmJYb+MJu2FXtbxzz9RsEv7SsxKMyH70nghiBKyO3Kb9afUf/p7UPPA7D/86yI4u1lseAh6j6+DSfBM/8FtzCDEarkin0Sk6RAAU+dcuqb8Tw4lQqbjysDdUeNsl4qTwEhIPCtxCBtZ7J/06pzVkiDOH9zKYdjVsgNkd4bBQ002NTIZLGdZlSYNpMmoA0P4jV4ucu3UE3MH3+uPPN/Yfw07Zx3d/7xgq+1Sf6FVTtF87j33AgMBAAECggEAL1IHUoD9iLX6kSbN5h42nXOHfVs+UKo4Ejz2iqPDFQmGyNp2qmKJlA4BVqHiW30MXRMOXbSXrNWAc+9dnUHaE6BLIcVvis3hQjbSi3W6dKU9CVLZZ5sELtT5lmA3VO6hViyh4eeT2O6F6hyEwaX6lHgy1Kb4KGT1UJ8q9Jy/R5a2jUZ6nsgtjSTQCIoZ28Fyc6V+M67KjKOp3js8tuekHvizlHDUQq5+nUxGqi3BFJQdKL/SVhX23wBC79RjX/xiotc7cy1oez8d0wBm2xMcws8KSGPDNA45RdRMq1UKfVr6MhraQbADp5bGg/CgJ/PkVTaYBO58/vsrzEeYohNTYQKBgQD+FjP20zOmXGcUqOd6dqvwDb3uvsvtmVXQGAzNtx1tFUaCEssUjGSILXPv/4n+1E52UX2SvWCi3S5/pdJenBNTS3VEhNQ0f9938VlgV6KjC9NyBZevv0Ck419QDoBM0IlpNmLuX5Bu7zzTfgiEwGnykTZc2sSfSYdV573idY4vhwKBgQDbSPD8fMwZuxYqWKmbCp4+uO4qhn96H5dVTq8yeEPrEnMR6dat/up+lv7hsuih6Yyx5En/YzYccb4y/u3bKWlbbS0JaXuOq4pm/O7VJUCCnveGr0aAfw3K1BlaT2KuCu+d22JuQwxb6m0jEVaRz/xtaeAK0jJeaWjI6eKOO6O6EQKBgQDuDC8zGDxTqE6K9JOqEHSVusovW2IDgJz9EQst0FkpUb11L9ZDsiSd0LfGKktHC/FWuJJFYFehPkfKH7kwGlmguBlt0GRfq03aOavar0AYMYQwVzz90tjlooAxqORbEyUqBWI6oh3XURxTqMiiVwLr11BvPibodSLuQOeNu0UzEQKBgQDEWE53NWMnEu5o4lSTH3HL5ZhTeXSBK/I8DyUuAqOFw1JHUuLHjv32dUspWiGUgGNbEXb0dqgH1gyg6sSdPTo6A8qUp99bQtu9FXLgZOebTa2C6kQtYBZAa6N7yqMoBWDCLqK26dASKRipkqe6DcoPSgKyjb1N7gJGC62w/1O9kQKBgGOpjIbbc0G3YYJnfC0G8mvh3FhDxUITRRIgCvLYQF1SfTKUHvE4FBDDH7T/G1P1791Cu5f9tIE+/dwheOVS04/rAr3gJxoPqrKHC7R4DJxR17wc2NBx/CHs8okLk0fcxk0b/Rc5mAm4ZVC8K8hpSzsq3v7WcHbPcEy7b//BcB6s
-----END PRIVATE KEY-----'
        );

        $samlRequestXml = \gzinflate(\base64_decode($request->query->get('SAMLRequest', ''), true) ?: '');
        $desContext     = new DeserializationContext();
        static::assertNotFalse($samlRequestXml);
        $desContext->getDocument()->loadXML($samlRequestXml);

        $authnRequestNode = $desContext->getDocument()->firstChild;
        static::assertNotNull($authnRequestNode);
        $authnRequest     = new AuthnRequest();
        $authnRequest->deserialize($authnRequestNode, $desContext);

        $processor = new AuthnRequestProcessor($configs, new SamlResponseGenerator($configs));
        $response  = $processor->process($authnRequest, $request);

        $assertion = $response->getFirstAssertion();
        static::assertNotNull($assertion);

        static::assertEquals('xyz', $assertion->getSubject()->getNameID()->getValue());
        static::assertEquals($authnRequest->getID(), $assertion->getSubject()->getFirstSubjectConfirmation()?->getSubjectConfirmationData()?->getInResponseTo());

        $conditions   = $assertion->getConditions();
        $audienceRest = $conditions?->getFirstAudienceRestriction();
        static::assertNotNull($audienceRest);

        static::assertEquals('urn:amazon:cognito:sp:us-east-1_eNuYdj18d', $audienceRest->getAllAudience()[0]);
    }
}
