<?php

/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\IdpBundle\Service;

use Cyber\IdpBundle\Entity\ServiceProvider;
use Cyber\IdpBundle\Entity\ServiceProviderRepoInterface;
use Cyber\IdpBundle\Event\FinalizeAssertionsEvent;
use Cyber\IdpBundle\Event\FinalizeResponseEvent;
use Cyber\IdpBundle\Service\AssertionProviderInterface;
use Cyber\IdpBundle\Service\SamlConfig;
use Cyber\IdpBundle\Service\SamlResponseGenerator;
use Cyber\IdpBundle\Service\SsoInitiator;
use LightSaml\Model\Assertion\AudienceRestriction;
use LightSaml\Model\Context\DeserializationContext;
use LightSaml\Model\Protocol\Response;
use PHPUnit\Framework\TestCase;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;

/**
 * @internal
 *
 * @coversNothing
 */
class SsoInitiatorTest extends TestCase
{
    public function testGenerateIdpInitiatedSso(): void
    {
        $provider          = $this->createMock(ServiceProvider::class);
        $assertionProvider = $this->createMock(AssertionProviderInterface::class);
        $assertionProvider->method('getUserNameId')->with($provider)->willReturn('xyz');

        $providerRepo = $this->createMock(ServiceProviderRepoInterface::class);
        $providerRepo->method('findByEntityId')->with('providerIdHere')
            ->willReturn($provider);
        $providerRepo->method('getAssertionProvider')->with($provider)->willReturn($assertionProvider);

        $configs = new SamlConfig(
            'abc',
            $providerRepo,
            '-----BEGIN CERTIFICATE-----
MIIDRTCCAi0CFGGhINyavdr3Va9VHRz9T2RI3ZTFMA0GCSqGSIb3DQEBCwUAMF4xCzAJBgNVBAYTAlVTMQswCQYDVQQIDAJQQTEPMA0GA1UEBwwGRWFzdG9uMSEwHwYDVQQKDBhJbnRlcm5ldCBXaWRnaXRzIFB0eSBMdGQxDjAMBgNVBAMMBWN5YmVyMCAXDTI0MTAwMTA0MDY1MloYDzIwNTIwMjE2MDQwNjUyWjBeMQswCQYDVQQGEwJVUzELMAkGA1UECAwCUEExDzANBgNVBAcMBkVhc3RvbjEhMB8GA1UECgwYSW50ZXJuZXQgV2lkZ2l0cyBQdHkgTHRkMQ4wDAYDVQQDDAVjeWJlcjCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoCggEBANmlY+IuiHc2+0Jfq7gjJ65mdcQmXq4aEG3mUoeK0dLFqR9WZAPUl9MmIZuy6C79JAY95sDUh1jKwN/tV8/XUAepBXsYsO9oJqo2heOVGYlhv4wm7YVe1vHPP1GwS/tKzEozIfvSeCGIErI7cpv1p9R/+ntQ88DsP/zrIji7WWx4CHqPr4NJ8Ez/wW3MIMRquSKfRKTpEABT51y6pvxPDiVCpuPKwN1R42yXipPASEg8K3EIG1nsn/TqnNWSIM4f3Mph2NWyA2R3hsFDTTY1MhksZ1mVJg2kyagDQ/iNXi5y7dQTcwff648839h/DTtnHd3/vGCr7VJ/oVVO0XzuPfcCAwEAATANBgkqhkiG9w0BAQsFAAOCAQEASkQIA/K9gkySPY0bOxaCmeyVV66RGojjTQhJ8oVY6q8pY5ICdGDw8erX3Sbj7dhGb8zV2zeuk2BPHHDYdnDwStNL0tz05A73czVkXW6SI2AHdMlUgTayRed3PoVAQSukS363fVeto2HRokrSkmvYwkQB45ds74t6QwrbPOkPiLhlOpeqL4wvYLSxdWjpow9j1MNZwJD7aWLr58qE0ASDFz+kWsIdryYh9HwfY0brv1VX8VeFj7JVvYf+Hq72ouV8GlkQKRLbPYKdD5GUcBIkZLCiGfWoiJFqq+gvsF7t3FMcefRzPIWb4c/6MJDvnAoKDxYHthtAu9vG2mK8ioESVQ==
-----END CERTIFICATE-----',
            '-----BEGIN PRIVATE KEY-----
MIIEvgIBADANBgkqhkiG9w0BAQEFAASCBKgwggSkAgEAAoIBAQDZpWPiLoh3NvtCX6u4IyeuZnXEJl6uGhBt5lKHitHSxakfVmQD1JfTJiGbsugu/SQGPebA1IdYysDf7VfP11AHqQV7GLDvaCaqNoXjlRmJYb+MJu2FXtbxzz9RsEv7SsxKMyH70nghiBKyO3Kb9afUf/p7UPPA7D/86yI4u1lseAh6j6+DSfBM/8FtzCDEarkin0Sk6RAAU+dcuqb8Tw4lQqbjysDdUeNsl4qTwEhIPCtxCBtZ7J/06pzVkiDOH9zKYdjVsgNkd4bBQ002NTIZLGdZlSYNpMmoA0P4jV4ucu3UE3MH3+uPPN/Yfw07Zx3d/7xgq+1Sf6FVTtF87j33AgMBAAECggEAL1IHUoD9iLX6kSbN5h42nXOHfVs+UKo4Ejz2iqPDFQmGyNp2qmKJlA4BVqHiW30MXRMOXbSXrNWAc+9dnUHaE6BLIcVvis3hQjbSi3W6dKU9CVLZZ5sELtT5lmA3VO6hViyh4eeT2O6F6hyEwaX6lHgy1Kb4KGT1UJ8q9Jy/R5a2jUZ6nsgtjSTQCIoZ28Fyc6V+M67KjKOp3js8tuekHvizlHDUQq5+nUxGqi3BFJQdKL/SVhX23wBC79RjX/xiotc7cy1oez8d0wBm2xMcws8KSGPDNA45RdRMq1UKfVr6MhraQbADp5bGg/CgJ/PkVTaYBO58/vsrzEeYohNTYQKBgQD+FjP20zOmXGcUqOd6dqvwDb3uvsvtmVXQGAzNtx1tFUaCEssUjGSILXPv/4n+1E52UX2SvWCi3S5/pdJenBNTS3VEhNQ0f9938VlgV6KjC9NyBZevv0Ck419QDoBM0IlpNmLuX5Bu7zzTfgiEwGnykTZc2sSfSYdV573idY4vhwKBgQDbSPD8fMwZuxYqWKmbCp4+uO4qhn96H5dVTq8yeEPrEnMR6dat/up+lv7hsuih6Yyx5En/YzYccb4y/u3bKWlbbS0JaXuOq4pm/O7VJUCCnveGr0aAfw3K1BlaT2KuCu+d22JuQwxb6m0jEVaRz/xtaeAK0jJeaWjI6eKOO6O6EQKBgQDuDC8zGDxTqE6K9JOqEHSVusovW2IDgJz9EQst0FkpUb11L9ZDsiSd0LfGKktHC/FWuJJFYFehPkfKH7kwGlmguBlt0GRfq03aOavar0AYMYQwVzz90tjlooAxqORbEyUqBWI6oh3XURxTqMiiVwLr11BvPibodSLuQOeNu0UzEQKBgQDEWE53NWMnEu5o4lSTH3HL5ZhTeXSBK/I8DyUuAqOFw1JHUuLHjv32dUspWiGUgGNbEXb0dqgH1gyg6sSdPTo6A8qUp99bQtu9FXLgZOebTa2C6kQtYBZAa6N7yqMoBWDCLqK26dASKRipkqe6DcoPSgKyjb1N7gJGC62w/1O9kQKBgGOpjIbbc0G3YYJnfC0G8mvh3FhDxUITRRIgCvLYQF1SfTKUHvE4FBDDH7T/G1P1791Cu5f9tIE+/dwheOVS04/rAr3gJxoPqrKHC7R4DJxR17wc2NBx/CHs8okLk0fcxk0b/Rc5mAm4ZVC8K8hpSzsq3v7WcHbPcEy7b//BcB6s
-----END PRIVATE KEY-----'
        );

        $dispatcher = $this->createMock(EventDispatcherInterface::class);

        $dispatcher->expects(static::exactly(2))->method('dispatch')->withConsecutive(
            [static::isInstanceOf(FinalizeAssertionsEvent::class)],
            [static::isInstanceOf(FinalizeResponseEvent::class)],
        )->willReturnCallback(static function ($event) {
            if ($event instanceof FinalizeAssertionsEvent) {
                // Create AudienceRestriction
                $audienceRestriction = new AudienceRestriction('event-test-restriction');

                // Add AudienceRestriction to Conditions
                $event->assertion->getConditions()?->addItem($audienceRestriction);
            }

            return $event;
        });

        $instance = new SsoInitiator($configs, new SamlResponseGenerator($configs), $dispatcher);

        $response = $instance->generateIdpInitiatedSso('providerIdHere');

        $desContext     = new DeserializationContext();
        static::assertNotFalse($response);
        $desContext->getDocument()->loadXML($response);

        $responseNode = $desContext->getDocument()->firstChild;
        static::assertNotNull($responseNode);
        $samlResponse     = new Response();
        $samlResponse->deserialize($responseNode, $desContext);

        static::assertEquals('abc', $samlResponse->getIssuer()?->getValue());

        $assertion = $samlResponse->getFirstAssertion();
        static::assertNotNull($assertion);
        static::assertEquals('xyz', $assertion->getSubject()->getNameID()->getValue());

        static::assertEquals('event-test-restriction', $assertion->getConditions()?->getFirstAudienceRestriction()?->getAllAudience()[0]);
    }
}
