<?php
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\Event\Subscriber;

use Doctrine\ORM\AbstractQuery;
use Doctrine\ORM\QueryBuilder;
use FOS\RestBundle\View\View;
use Knp\Component\Pager\PaginatorInterface;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpKernel\Event\ViewEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use UnexpectedValueException;

class PaginationSubscriber implements EventSubscriberInterface
{
    public const ATTRIBUTE = 'cy-pagination';

    /** @var RequestStack */
    private $stack;

    /** @var PaginatorInterface */
    private $paginator;

    /**
     * @var array<string, int>
     */
    private $defaults;

    /**
     * YATSPaginator constructor.
     *
     * @param RequestStack       $stack
     * @param PaginatorInterface $paginator
     * @param array<string, int> $defaults
     */
    public function __construct(RequestStack $stack, PaginatorInterface $paginator, array $defaults)
    {
        $this->stack     = $stack;
        $this->paginator = $paginator;
        $this->defaults  = $defaults;
    }

    /**
     * @return array<mixed>
     */
    public static function getSubscribedEvents(): array
    {
        return [
            KernelEvents::VIEW => [['paginateResponse', 1000]],
        ];
    }

    public function paginateResponse(ViewEvent $event): void
    {
        $wrapper = null;
        $result  = $event->getControllerResult();
        if ($result instanceof View) {
            $wrapper = $result;
            $result  = $wrapper->getData();
        }

        $request = $event->getRequest();

        // if controller returned query builder we assume we want to paginate it
        if (!$result instanceof QueryBuilder
            && !$result instanceof AbstractQuery
            && !$request->attributes->has(self::ATTRIBUTE)) {
            return;
        }

        $result = $this->paginate($request, $result);

        $event->setControllerResult($this->wrap($result, $wrapper));
    }

    /**
     * @param Request                    $currentRequest
     * @param AbstractQuery|QueryBuilder $queryBuilder
     *
     * @return \Knp\Component\Pager\Pagination\PaginationInterface<mixed>
     */
    private function paginate(
        Request $currentRequest,
        $queryBuilder
    ): \Knp\Component\Pager\Pagination\PaginationInterface {
        $masterRequest = $this->stack->getMasterRequest();

        if (null === $masterRequest) {
            // should never happen we can't have this event without masterRequest
            throw new UnexpectedValueException('No requests in stack');
        }

        $paginationAttributes = $currentRequest->attributes->get(self::ATTRIBUTE, []);
        $options              = \array_merge($this->defaults, $paginationAttributes);

        $page = $masterRequest->query->getInt('page', $options['default_page']);
        $page = \max($page, 1);

        $limit = $masterRequest->query->getInt('page_size', $options['default_page_size']);

        // adjust to allowed limits
        $limit = \max($limit, $options['min_page_size']);
        $limit = \min($limit, $options['max_page_size']);

        return $this->paginator->paginate($queryBuilder, $page, $limit);
    }

    /**
     * @template T
     *
     * @param T         $result
     * @param null|View $wrapper
     *
     * @return T|View
     */
    private function wrap($result, $wrapper)
    {
        if ($wrapper instanceof View) {
            return $wrapper->setData($result);
        }

        return $result;
    }
}
