<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle\Service;

use Aws\Exception\AwsException;
use Aws\Ses\SesClient;
use Swift_Events_EventListener;
use Swift_Mime_SimpleMessage;
use Swift_Transport;
use Swift_TransportException;

class AwsSesMailerTransport implements Swift_Transport
{
    /** @var SesClient */
    private $client;

    /**
     * @param SesClient $client
     */
    public function __construct(SesClient $client)
    {
        $this->client = $client;
    }

    /**
     * {@inheritdoc}
     */
    public function isStarted(): bool
    {
        return true;
    }

    /**
     * {@inheritdoc}
     */
    public function start(): void
    {
        // nothing to do here
    }

    /**
     * {@inheritdoc}
     */
    public function stop(): void
    {
        // nothing to do here
    }

    /**
     * {@inheritdoc}
     */
    public function ping(): bool
    {
        // should do we need to ping for SES servers to check for them being alive?
        return true;
    }

    /**
     * {@inheritdoc}
     *
     * @throws Swift_TransportException
     */
    public function send(Swift_Mime_SimpleMessage $message, &$failedRecipients = null): int
    {
        $from      = $this->getReversePath($message);
        $toEmails  = (array) $message->getTo();
        $ccEmails  = (array) $message->getCc();
        $bccEmails = (array) $message->getBcc();
        $allEmails = \array_merge($toEmails, $ccEmails, $bccEmails);
        $rawData   = $message->toString();

        if (null === $from) {
            throw new Swift_TransportException('Could not determine message sender address');
        }

        try {
            $result = $this->client->sendRawEmail([
                'Source'       => $from,
                'Destinations' => \array_keys($allEmails),
                'RawMessage'   => [
                    'Data' => $rawData,
                ],
            ]);

            // SES generates its own id which we want to save in the message for reference.
            $message->setId($result['MessageId'] . '@email.amazonses.com');

            return \count($toEmails) + \count($ccEmails) + \count($bccEmails);
        } catch (AwsException $e) {
            throw new Swift_TransportException('Failed to send message', 0, $e);
        }
    }

    /**
     * {@inheritdoc}
     */
    public function registerPlugin(Swift_Events_EventListener $plugin): void
    {
        // we don't support plugins
    }

    /**
     * Determine the best-use reverse path for this message.
     *
     * Borrowed from \Swift_Transport_AbstractSmtpTransport
     */
    protected function getReversePath(Swift_Mime_SimpleMessage $message): ?string
    {
        $return = $message->getReturnPath();
        /** @var array<string> $sender */
        $sender = $message->getSender();
        $from   = $message->getFrom();
        $path   = null;
        if (!empty($return)) {
            $path = $return;
        } elseif (!empty($sender)) {
            // Don't use array_keys
            \reset($sender); // Reset Pointer to first pos
            $path = \key($sender); // Get key
        } elseif (!empty($from)) {
            \reset($from); // Reset Pointer to first pos
            $path = \key($from); // Get key
        }

        return $path;
    }
}
