<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle;

use Cyber\MiscBundle\Exception\SheetProcessorException;

abstract class SpreadsheetDataProcessor
{
    /**
     * If set and there are extra cells in a row, they will be silently trimmed. If not set, exception will be raised.
     */
    public const OPT_IGNORE_EXTRA_COLUMNS = 'ignore-extra-columns';

    /**
     * If set and the there less columns than headings, extras will be added as empty strings, else exception is
     * raised.
     */
    public const OPT_IGNORE_MISSING_COLUMNS = 'ignore-missing-columns';

    /**
     * Returns the map of column names to key values they should be mapped to.
     *
     * For example if your sheet contains column "Patient Id" and you want to map it to just "id", then you would return
     * array ['Patient Id' => 'id'].
     *
     * Why do this mapping? Well in case your sheet headers change all you have to do is update your mapping array
     * in this function, while the rest of the code can remain unchanged as it will reference the mapped column name.
     *
     * @param array<mixed> $options arbitrary options that you can pass during iteration phase and they will be provided here.
     *                              In case you have variations in different sheets but all need to be handled by the same
     *                              fetcher you can return different mapping based on the options you pass.
     *
     * @return string[]
     */
    abstract protected function getColumnMapping(array $options): array;

    /**
     * Allows for additional mapping of each data row.
     *
     * If you wish the iterator to return objects instead of array of data you should override this function and
     * map the incoming $data onto and object.
     *
     *
     *
     * @param array<mixed> $data
     * @param int          $line    the line number in the sheet that is being processed. You can use this for error logging
     *                              and such.
     * @param array<mixed> $options arbitrary options that you can pass during iteration phase and they will be provided here.
     *                              You can use these to customize the mapping if necessary.
     *
     * @return mixed by default returns the array of data unless overridden by extending class
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    protected function mapRow(array &$data, int $line, array $options)
    {
        return $data;
    }

    /**
     * The first returned row of iterator must contain column headings.
     *
     * @param \Iterator<null|int, array<mixed>> $rowIterator the iterator over row data
     * @param array<mixed>                      $options
     *
     * @throws SheetProcessorException
     *
     * @return \Generator<array<mixed>>
     */
    public function process(\Iterator $rowIterator, array $options = []): \Generator
    {
        $rowIterator->rewind();
        if (!$rowIterator->valid()) {
            throw new SheetProcessorException('No data in the sheet. At least headers should be present.');
        }

        $line     = 1; // process the header row first
        $headRow  = $rowIterator->current();
        $headRow  = $this->mapColumns($headRow, $options);
        $colCount = \count($headRow);

        // now loop through rest of the rows
        $rowIterator->next();
        while ($rowIterator->valid()) {
            ++$line;
            $row = $rowIterator->current();
            $this->prepareRow($row, $colCount, $line, $options);
            $data = @\array_combine($headRow, $row);

            yield $this->mapRow($data, $line, $options);
            $rowIterator->next();
        }
    }

    /**
     * @param array<mixed> $headerRow
     * @param array<mixed> $options
     *
     * @throws SheetProcessorException
     *
     * @return array<mixed> $map
     */
    private function mapColumns(array &$headerRow, array $options): array
    {
        $columnFormat = $this->getColumnMapping($options);
        $validate     = \array_diff(\array_keys($columnFormat), $headerRow);

        if (0 !== \count($validate)) {
            throw new SheetProcessorException('Report format does not match to expected.
            Could not find the following expected columns: ' . \implode(', ', \array_values($validate)));
        }

        $map = [];

        foreach ($headerRow as $key => $value) {
            if (isset($columnFormat[$value])) {
                $map[] = $columnFormat[$value];
                continue;
            }
            // map any extra columns to unknown columns.
            $map[] = 'unknown_column_' . $key;
        }

        return $map;
    }

    /**
     * @param array<mixed> $row
     * @param int          $colCount
     * @param int          $line
     * @param array<mixed> $options
     *
     * @throws SheetProcessorException
     */
    private function prepareRow(&$row, $colCount, $line, &$options): void
    {
        $actualCount = \count($row);
        if ($colCount === $actualCount) {
            // all checks out
            return;
        }

        if ($colCount < $actualCount && isset($options[self::OPT_IGNORE_EXTRA_COLUMNS])) {
            \array_splice($row, $colCount); // remove all elements from the end

            return;
        }

        if ($colCount > $actualCount && isset($options[self::OPT_IGNORE_MISSING_COLUMNS])) {
            for ($i = $colCount - $actualCount; $i > 0; --$i) {
                $row[] = '';
            }

            return;
        }

        throw new SheetProcessorException(
            \sprintf(
                'The number of columns at line %d did not match the number of headers. Expected %d, but got %d.',
                $line,
                $colCount,
                $actualCount
            )
        );
    }
}
