<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Cyber\MiscBundle;

use Box\Spout\Common\Entity\Row;
use Box\Spout\Reader\Common\Creator\ReaderFactory;
use Box\Spout\Reader\SheetInterface;
use Iterator;
use RuntimeException;

/**
 * A helper class to easily produce iterator required for SpreadsheetDataProcessor(s).
 *
 * For this class to work you need to add composer dependency for box/spout 3+
 *
 * @implements Iterator<null|int, array<mixed>>
 */
class SpreadsheetIterator implements Iterator
{
    public const TYPE_CSV = 'CSV';

    public const TYPE_ODS = 'ODS';

    public const TYPE_XLSX = 'XLSX';

    /** @var string */
    private $sourceFile;

    /**
     * @var null|string
     */
    private $sheetType;

    /**
     * @var null|\Box\Spout\Reader\ReaderInterface
     */
    private $reader;

    /**
     * @var null|\Box\Spout\Reader\IteratorInterface<int, Row>
     */
    private $rowIterator;

    /**
     * @param string      $sourceFile
     * @param null|string $sheetType  one of TYPE_* constants
     */
    public function __construct(string $sourceFile, string $sheetType = null)
    {
        $this->throwOnMissingClass(ReaderFactory::class);
        $this->sourceFile = $sourceFile;
        $this->sheetType  = $sheetType;
    }

    /**
     * {@inheritdoc}
     */
    public function current(): ?array
    {
        if (!$this->rowIterator) {
            return null;
        }

        /** @var Row $row */
        $row = $this->rowIterator->current();

        return $row->toArray();
    }

    /**
     * {@inheritdoc}
     */
    public function next(): void
    {
        if (!$this->rowIterator) {
            return;
        }

        $this->rowIterator->next();
    }

    /**
     * {@inheritdoc}
     */
    public function key()
    {
        if (!$this->rowIterator) {
            return null;
        }

        // for some reason box spout decided for keys to be 1 based instead of 0 so convert that to 0 based,
        // since that makes everything else we ever write make more sense.
        return $this->rowIterator->key() - 1;
    }

    /**
     * {@inheritdoc}
     */
    public function valid(): bool
    {
        if (!$this->rowIterator) {
            return false;
        }

        if ($this->rowIterator->valid()) {
            return true;
        }

        // when we reach the end iterator becomes invalid, close the reader to release memory.
        if ($this->reader) { // reader should never be null if we reach this point, but stan complains so just check it
            $this->reader->close();
        }

        return false;
    }

    /**
     * {@inheritdoc}
     *
     * @throws \Box\Spout\Common\Exception\UnsupportedTypeException
     * @throws \Box\Spout\Common\Exception\IOException
     * @throws \Box\Spout\Reader\Exception\ReaderNotOpenedException
     * @SuppressWarnings(PHPMD.StaticAccess)
     */
    public function rewind(): void
    {
        if ($this->reader) {
            $this->reader->close();
            $this->rowIterator = null;
        }

        $reader = $this->reader = (null === $this->sheetType ? ReaderFactory::createFromFile($this->sourceFile) : ReaderFactory::createFromType($this->sheetType));

        $reader->open($this->sourceFile);

        $sheetIterator = $reader->getSheetIterator();
        $sheetIterator->rewind();
        /** @var SheetInterface $sheet */
        $sheet = $sheetIterator->current(); // we only care about first sheet

        $this->rowIterator = $sheet->getRowIterator();
        $this->rowIterator->rewind();
    }

    public function throwOnMissingClass(string $class): void
    {
        if (!\class_exists($class)) {
            throw new RuntimeException(
                'box/spout dependency was not found. Require it in composer to use this iterator'
            );
        }
    }
}
