<?php declare(strict_types=1);
/**
 * This file is subject to the terms and conditions defined in file 'LICENSE', which is part of this source code
 * package. If the file is missing a copy can be found at:
 * https://gitlab.cybercoder.site/vj/policies-procedures-standards/blob/master/licensing/CYBER-LICENSE.
 */

namespace Tests\Cyber\MiscBundle;

use ArrayIterator;
use Cyber\MiscBundle\SpreadsheetDataProcessor;
use PHPUnit\Framework\TestCase;
use stdClass;

/**
 * @covers \Cyber\MiscBundle\SpreadsheetDataProcessor
 */
class SpreadsheetDataProcessorTest extends TestCase
{
    /**
     * @var SpreadsheetDataProcessor
     */
    private $instance;

    /**
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    protected function setUp(): void
    {
        $this->instance = new class() extends SpreadsheetDataProcessor {
            /**
             * {@inheritdoc}
             */
            protected function getColumnMapping(array $options): array
            {
                return [
                    'col 1'     => 'id',
                    'other col' => 'data',
                ];
            }

            /**
             * {@inheritdoc}
             */
            protected function mapRow(array &$data, int $line, array $options)
            {
                // convert to stdClass
                return (object) $data;
            }
        };
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException never
     */
    public function testProcessSuccess(): void
    {
        $data = [
            ['col 1', 'other col'],
            ['1', '2'],
            ['3', '4'],
        ];

        /** @var stdClass[] $results */
        $results = \iterator_to_array($this->instance->process(new ArrayIterator($data)));

        $this->assertCount(2, $results);
        $this->assertContainsOnlyInstancesOf(\stdClass::class, $results);
        $this->assertEquals('1', $results[0]->id);
        $this->assertEquals('2', $results[0]->data);
        $this->assertEquals('3', $results[1]->id);
        $this->assertEquals('4', $results[1]->data);
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException always
     * @expectedException \Cyber\MiscBundle\Exception\SheetProcessorException
     * @expectedExceptionMessage Report format does not match to expected
     */
    public function testProcessMissingOrInvalidColumns(): void
    {
        $data = [
            ['col 1', 'wrong column'],
            ['1', '2'],
            ['3', '4'],
        ];

        \iterator_to_array($this->instance->process(new ArrayIterator($data)));
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException never
     */
    public function testProcessExtraColumns(): void
    {
        $data = [
            ['col 1', 'other col', 'extra column'],
            ['1', '2', 'a'],
            ['3', '4', 'b'],
        ];

        /** @var stdClass[] $results */
        $results = \iterator_to_array($this->instance->process(new ArrayIterator($data)));

        $this->assertCount(2, $results);
        $this->assertContainsOnlyInstancesOf(stdClass::class, $results);
        $this->assertEquals('a', $results[0]->unknown_column_2);
        $this->assertEquals('b', $results[1]->unknown_column_2);
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException never
     */
    public function testProcessIgnoreExtraValuesFlag(): void
    {
        $data = [
            ['col 1', 'other col'],
            ['1', '2', 'a'],
            ['3', '4', 'b'],
        ];

        $results = \iterator_to_array($this->instance->process(
            new ArrayIterator($data),
            [SpreadsheetDataProcessor::OPT_IGNORE_EXTRA_COLUMNS => true]
        ));

        $this->assertCount(2, $results);
        $this->assertContainsOnlyInstancesOf(stdClass::class, $results);
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException never
     */
    public function testProcessIgnoreMissingValuesFlag(): void
    {
        $data = [
            ['col 1', 'other col'],
            ['1'],
            ['3'],
        ];

        /** @var stdClass[] $results */
        $results = \iterator_to_array($this->instance->process(
            new ArrayIterator($data),
            [SpreadsheetDataProcessor::OPT_IGNORE_MISSING_COLUMNS => true]
        ));

        $this->assertCount(2, $results);
        $this->assertContainsOnlyInstancesOf(stdClass::class, $results);
        $this->assertEquals('1', $results[0]->id);
        $this->assertEquals('', $results[0]->data);
        $this->assertEquals('3', $results[1]->id);
        $this->assertEquals('', $results[1]->data);
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException always
     * @expectedException \Cyber\MiscBundle\Exception\SheetProcessorException
     * @expectedExceptionMessage Expected 2, but got 3
     */
    public function testProcessFailOnExtraValues(): void
    {
        $data = [
            ['col 1', 'other col'],
            ['1', '2', 'a'],
            ['3', '4', 'b'],
        ];

        \iterator_to_array($this->instance->process(new ArrayIterator($data)));
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException always
     * @expectedException \Cyber\MiscBundle\Exception\SheetProcessorException
     * @expectedExceptionMessage Expected 2, but got 1
     */
    public function testProcessFailOnMissingValues(): void
    {
        $data = [
            ['col 1', 'other col'],
            ['1'],
            ['3'],
        ];

        \iterator_to_array($this->instance->process(new ArrayIterator($data)));
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException always
     * @expectedException \Cyber\MiscBundle\Exception\SheetProcessorException
     * @expectedExceptionMessage No data in the sheet
     */
    public function testProcessEmptyIterator(): void
    {
        $data = [];

        \iterator_to_array($this->instance->process(new ArrayIterator($data)));
    }

    /**
     * @throws \Cyber\MiscBundle\Exception\SheetProcessorException
     * @SuppressWarnings(PHPMD.UnusedFormalParameter)
     */
    public function testDefaultMap(): void
    {
        $this->instance = new class() extends SpreadsheetDataProcessor {
            /**
             * {@inheritdoc}
             */
            protected function getColumnMapping(array $options): array
            {
                return [
                    'col 1'     => 'id',
                    'other col' => 'data',
                ];
            }
        };

        $data = [
            ['col 1', 'other col'],
            ['1', '2'],
            ['3', '4'],
        ];

        $results = \iterator_to_array($this->instance->process(new ArrayIterator($data)));
        $this->assertCount(2, $results);
        $this->assertEquals('1', $results[0]['id']);
        $this->assertEquals('2', $results[0]['data']);
        $this->assertEquals('3', $results[1]['id']);
        $this->assertEquals(4, $results[1]['data']);
    }
}
